<?php
/**
 * ============================================================
 * أوامر الزبون (User Commands)
 * ============================================================
 * 
 * يحتوي جميع أوامر الزبون العادي:
 * - تفعيل أكواد التفعيل ACT-...
 * - طلب أكواد التحقق (get_code)
 * - عرض الحسابات المملوكة
 * 
 * ============================================================
 * ملاحظات:
 * ============================================================
 * - الأوامر الحساسة تتطلب اشتراك بالقناة
 * - يتم فحص حدود طلب أكواد التحقق قبل التوليد
 * - يتم فحص صلاحية الاشتراك (انتهاء/حظر)
 */

class UserCommands
{
    /**
     * معالجة الأمر الوارد
     */
    public static function handle(string $text, int $fromId, int $chatId, array $message, array $user): void
    {
        // تحليل الأمر
        $parts = preg_split('/\s+/', $text, 2);
        $command = strtolower($parts[0]);
        $args = $parts[1] ?? '';
        
        switch ($command) {
            // طلب كود تحقق
            case 'get_code':
            case '/get_code':
                self::cmdGetCode($chatId, $fromId, $user, $args);
                break;
            
            // معلومات حساب
            case '/account_info':
                self::cmdAccountInfo($chatId, $user, $args);
                break;
            
            default:
                Telegram::sendMessage($chatId, "❓ أمر غير معروف.\n\nاستخدم /help لعرض الأوامر.");
        }
    }
    
    // ================================================================
    // أوامر عامة
    // ================================================================
    
    /**
     * رسالة الترحيب
     */
    public static function cmdStart(int $chatId, array $user): void
    {
        // تحديد السوبر موزع المصدر (إن وجد)
        $superdistId = self::getUserSuperDistId($user['id']);
        
        $text = Replies::getForSuperDist('WELCOME_USER', $superdistId, [
            'name' => $user['first_name'] ?? 'صديقنا',
        ]);
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض المساعدة
     */
    public static function cmdHelp(int $chatId): void
    {
        $text = Replies::get('HELP_USER');
        Telegram::sendMessage($chatId, $text);
    }
    
    // ================================================================
    // الحسابات
    // ================================================================
    
    /**
     * عرض حساباتي
     */
    public static function cmdMyAccounts(int $chatId, array $user): void
    {
        $accounts = Db::fetchAll(
            "SELECT ua.*, a.slug, a.name
             FROM user_accounts ua
             JOIN accounts a ON a.id = ua.account_id
             WHERE ua.user_id = ?
             ORDER BY ua.status DESC, ua.access_expires_at DESC",
            [$user['id']]
        );
        
        if (empty($accounts)) {
            Telegram::sendMessage($chatId, "📦 لا تملك أي حسابات حالياً.\n\nأرسل كود التفعيل لتفعيل حساب.");
            return;
        }
        
        $text = "📦 <b>حساباتك</b>\n\n";
        
        foreach ($accounts as $acc) {
            $isActive = $acc['status'] === 'active';
            
            // التحقق من الانتهاء
            if ($isActive && $acc['access_type'] === 'timed' && $acc['access_expires_at']) {
                if (strtotime($acc['access_expires_at']) < time()) {
                    $isActive = false;
                }
            }
            
            $icon = $isActive ? '🟢' : '🔴';
            
            $text .= "{$icon} <b>{$acc['name']}</b>\n";
            $text .= "   └ Slug: <code>{$acc['slug']}</code>\n";
            
            if ($acc['access_type'] === 'lifetime') {
                $text .= "   └ النوع: مدى الحياة ✨\n";
            } else {
                if ($acc['access_expires_at']) {
                    $remaining = formatRemainingDays($acc['access_expires_at']);
                    $text .= "   └ متبقي: {$remaining}\n";
                }
            }
            
            $text .= "\n";
        }
        
        $text .= "─────────────────\n";
        $text .= "لطلب كود تحقق: <code>get_code [slug]</code>";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * معلومات حساب
     */
    public static function cmdAccountInfo(int $chatId, array $user, string $accountSlug): void
    {
        if (empty($accountSlug)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام: <code>/account_info [slug]</code>");
            return;
        }
        
        $account = getAccountBySlug(strtolower($accountSlug));
        if (!$account) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود.");
            return;
        }
        
        // جلب اشتراك المستخدم
        $ua = Db::fetchOne(
            "SELECT * FROM user_accounts WHERE user_id = ? AND account_id = ?",
            [$user['id'], $account['id']]
        );
        
        if (!$ua) {
            Telegram::sendMessage($chatId, "❌ أنت لا تملك هذا الحساب.");
            return;
        }
        
        $text = "📦 <b>معلومات حسابك</b>\n\n";
        $text .= "• الاسم: {$account['name']}\n";
        $text .= "• Slug: <code>{$account['slug']}</code>\n";
        $text .= "• الحالة: " . ($ua['status'] === 'active' ? '🟢 نشط' : '🔴 غير نشط') . "\n";
        
        if ($ua['access_type'] === 'lifetime') {
            $text .= "• النوع: مدى الحياة ✨\n";
        } else {
            $text .= "• النوع: مؤقت\n";
            if ($ua['access_expires_at']) {
                $remaining = formatRemainingDays($ua['access_expires_at']);
                $text .= "• ينتهي: " . formatDate($ua['access_expires_at']) . " ({$remaining})\n";
            }
        }
        
        $text .= "• فُعّل: " . formatDate($ua['activated_at']) . "\n";
        
        // إحصائيات الاستخدام
        $todayUsage = Db::count('verification_logs', 'user_id = ? AND account_id = ? AND DATE(requested_at) = CURDATE()', [$user['id'], $account['id']]);
        $totalUsage = Db::count('verification_logs', 'user_id = ? AND account_id = ?', [$user['id'], $account['id']]);
        
        $text .= "\n📊 الاستخدام:\n";
        $text .= "• اليوم: {$todayUsage} طلب\n";
        $text .= "• الإجمالي: {$totalUsage} طلب\n";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    // ================================================================
    // طلب كود التحقق
    // ================================================================
    
    /**
     * طلب كود تحقق
     * الصيغة: get_code <account_slug>
     */
    public static function cmdGetCode(int $chatId, int $fromId, array $user, string $accountSlug): void
    {
        if (empty($accountSlug)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام:\n<code>get_code [account_slug]</code>\n\nمثال:\n<code>get_code vip-acc</code>\n\nاستخدم /myaccounts لعرض حساباتك.");
            return;
        }
        
        // فحص الاشتراك بالقناة
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        
        self::processGetCode($chatId, $user, strtolower($accountSlug));
    }
    
    /**
     * طلب كود تحقق عبر الـ slug مباشرة
     */
    public static function handleGetCodeBySlug(string $slug, int $fromId, int $chatId, array $user): void
    {
        self::processGetCode($chatId, $user, strtolower($slug));
    }
    
    /**
     * معالجة طلب كود التحقق
     * 
     * ============================================================
     * حماية من Race Condition:
     * ============================================================
     * نستخدم FOR UPDATE على سجل المستخدم لمنع الطلبات المتزامنة
     * من تجاوز الحدود. الفحص والتسجيل يحدثان في نفس الـ transaction.
     */
    private static function processGetCode(int $chatId, array $user, string $accountSlug): void
    {
        // جلب الحساب
        $account = getAccountBySlug($accountSlug);
        if (!$account) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود: {$accountSlug}");
            return;
        }
        
        // التحقق من أن الحساب نشط
        if ($account['status'] !== 'active') {
            Telegram::sendMessage($chatId, "❌ هذا الحساب غير نشط حالياً.");
            return;
        }
        
        try {
            Db::beginTransaction();
            
            // ============================================================
            // قفل سجل المستخدم لمنع الطلبات المتزامنة من نفس المستخدم
            // ============================================================
            $ua = Db::fetchOne(
                "SELECT ua.*, sd.id as superdist_id
                 FROM user_accounts ua
                 LEFT JOIN super_distributors sd ON sd.id = ua.superdist_id
                 WHERE ua.user_id = ? AND ua.account_id = ?
                 FOR UPDATE",
                [$user['id'], $account['id']]
            );
            
            if (!$ua) {
                Db::rollback();
                $superdistId = self::getUserSuperDistId($user['id']);
                Telegram::sendMessage($chatId, Replies::getForSuperDist('ERROR_ACCOUNT_NOT_OWNED', $superdistId));
                return;
            }
            
            // التحقق من حالة الاشتراك
            if ($ua['status'] !== 'active') {
                Db::rollback();
                Telegram::sendMessage($chatId, "❌ اشتراكك في هذا الحساب غير نشط.");
                return;
            }
            
            // التحقق من انتهاء الصلاحية
            if ($ua['access_type'] === 'timed' && $ua['access_expires_at']) {
                if (strtotime($ua['access_expires_at']) < time()) {
                    // تحديث الحالة إلى منتهي
                    Db::update('user_accounts', ['status' => 'expired'], 'id = ?', [$ua['id']]);
                    Db::commit();
                    
                    $superdistId = $ua['superdist_id'];
                    Telegram::sendMessage($chatId, Replies::getForSuperDist('ERROR_ACCOUNT_EXPIRED', $superdistId));
                    return;
                }
            }
            
            // التحقق من الحدود (بعد القفل - بيانات محدثة)
            $superdistId = $ua['superdist_id'];
            $limitCheck = Limits::checkVerificationLimit($user['id'], $account['id'], $superdistId);
            
            if (!$limitCheck['allowed']) {
                Db::rollback();
                self::sendLimitError($chatId, $limitCheck, $superdistId);
                Logger::limitExceeded($user['id'], $account['id'], $limitCheck['limit_type'] ?? 'unknown', $limitCheck['limit'] ?? 0, $limitCheck['used'] ?? 0);
                return;
            }
            
            // توليد كود التحقق
            $verificationCode = CodeGenerator::generateVerificationCode($account['secret_key']);
            $remainingSeconds = CodeGenerator::getVerificationCodeRemainingSeconds();
            
            // تسجيل الطلب (داخل الـ transaction)
            Db::insert('verification_logs', [
                'user_id'           => $user['id'],
                'account_id'        => $account['id'],
                'superdist_id'      => $superdistId,
                'verification_code' => $verificationCode,
            ]);
            
            Db::commit();
            
            Logger::verificationRequested($user['id'], $account['id'], $verificationCode);
            
            // إرسال الكود
            $text = Replies::getForSuperDist('SUCCESS_VERIFICATION_CODE', $superdistId, [
                'account_name' => $account['name'],
                'code'         => $verificationCode,
                'remaining'    => $remainingSeconds,
            ]);
            
            Telegram::sendMessage($chatId, $text);
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Failed to process get_code', ['error' => $e->getMessage()]);
            Telegram::sendMessage($chatId, "❌ فشل طلب الكود. حاول مرة أخرى.");
        }
    }
    
    // ================================================================
    // تفعيل الأكواد
    // ================================================================
    
    /**
     * معالجة كود تفعيل ACT-...
     */
    public static function handleActivationCode(string $code, int $fromId, int $chatId, array $user): void
    {
        $code = strtoupper(trim($code));
        
        // جلب الكود
        $actCode = Db::fetchOne(
            "SELECT ac.*, a.slug, a.name as account_name, a.secret_key,
                    sd.id as superdist_id, sd.display_name as superdist_name
             FROM activation_codes ac
             JOIN accounts a ON a.id = ac.account_id
             JOIN super_distributors sd ON sd.id = ac.superdist_id
             WHERE ac.code = ?",
            [$code]
        );
        
        if (!$actCode) {
            Telegram::sendMessage($chatId, Replies::get('ERROR_INVALID_CODE'));
            return;
        }
        
        // التحقق من حالة الكود
        if ($actCode['status'] !== 'unused') {
            $errorKey = match ($actCode['status']) {
                'used'     => 'ERROR_CODE_USED',
                'expired'  => 'ERROR_CODE_EXPIRED',
                'disabled' => 'ERROR_CODE_DISABLED',
                default    => 'ERROR_INVALID_CODE',
            };
            Telegram::sendMessage($chatId, Replies::get($errorKey));
            return;
        }
        
        // التحقق من الصلاحية
        if ($actCode['expires_at'] && strtotime($actCode['expires_at']) < time()) {
            Db::update('activation_codes', ['status' => 'expired'], 'id = ?', [$actCode['id']]);
            Telegram::sendMessage($chatId, Replies::get('ERROR_CODE_EXPIRED'));
            return;
        }
        
        // تفعيل الكود
        try {
            Db::beginTransaction();
            
            $accountId = $actCode['account_id'];
            $superdistId = $actCode['superdist_id'];
            $accessType = $actCode['access_type'];
            $accessDays = $actCode['access_days'] ?? 0;
            
            // حساب تاريخ الانتهاء
            $expiresAt = null;
            if ($accessType === 'timed' && $accessDays > 0) {
                $expiresAt = date('Y-m-d H:i:s', strtotime("+{$accessDays} days"));
            }
            
            // التحقق من وجود اشتراك سابق
            $existingUa = Db::fetchOne(
                "SELECT * FROM user_accounts WHERE user_id = ? AND account_id = ?",
                [$user['id'], $accountId]
            );
            
            if ($existingUa) {
                // ============================================================
                // قاعدة ذهبية: لا يُخفَّض lifetime أبداً!
                // ============================================================
                
                if ($existingUa['access_type'] === 'lifetime') {
                    // المستخدم لديه lifetime بالفعل
                    if ($accessType === 'lifetime') {
                        // تجديد lifetime - فقط تحديث المصدر
                        Db::update('user_accounts', [
                            'status'            => 'active',
                            'superdist_id'      => $superdistId,
                            'activation_code_id'=> $actCode['id'],
                        ], 'id = ?', [$existingUa['id']]);
                    }
                    // كود timed على اشتراك lifetime: نقبل الكود لكن لا نُخفّض!
                    // الاشتراك يبقى lifetime كما هو
                    $accessType = 'lifetime'; // للعرض الصحيح
                    $expiresAt = null;
                    
                } elseif ($accessType === 'lifetime') {
                    // ترقية من timed إلى lifetime
                    Db::update('user_accounts', [
                        'access_type'       => 'lifetime',
                        'access_expires_at' => null,
                        'status'            => 'active',
                        'superdist_id'      => $superdistId,
                        'activation_code_id'=> $actCode['id'],
                    ], 'id = ?', [$existingUa['id']]);
                    
                } else {
                    // كلاهما timed - تمديد الفترة
                    $newExpiresAt = $expiresAt;
                    
                    // إذا كان الاشتراك لا يزال نشطاً، أضف الأيام
                    if ($existingUa['access_expires_at'] && strtotime($existingUa['access_expires_at']) > time()) {
                        $newExpiresAt = date('Y-m-d H:i:s', strtotime($existingUa['access_expires_at'] . " +{$accessDays} days"));
                    }
                    
                    Db::update('user_accounts', [
                        'access_type'       => 'timed',
                        'access_expires_at' => $newExpiresAt,
                        'status'            => 'active',
                        'superdist_id'      => $superdistId,
                        'activation_code_id'=> $actCode['id'],
                    ], 'id = ?', [$existingUa['id']]);
                    
                    $expiresAt = $newExpiresAt;
                }
            } else {
                // إنشاء اشتراك جديد
                Db::insert('user_accounts', [
                    'user_id'           => $user['id'],
                    'account_id'        => $accountId,
                    'superdist_id'      => $superdistId,
                    'activation_code_id'=> $actCode['id'],
                    'access_type'       => $accessType,
                    'access_days'       => $accessDays ?: null,
                    'access_expires_at' => $expiresAt,
                    'status'            => 'active',
                ]);
            }
            
            // تحديث حالة الكود
            Db::update('activation_codes', [
                'status'          => 'used',
                'used_by_user_id' => $user['id'],
                'used_at'         => date('Y-m-d H:i:s'),
            ], 'id = ?', [$actCode['id']]);
            
            Db::commit();
            
            // تسجيل الحدث
            Logger::accountActivated($user['id'], $accountId, $code, $superdistId);
            
            // إعداد معلومات الانتهاء
            $expiryInfo = '';
            if ($accessType === 'lifetime') {
                $expiryInfo = 'مدى الحياة ✨';
            } elseif ($expiresAt) {
                $expiryInfo = 'ينتهي: ' . formatDate($expiresAt) . ' (' . formatRemainingDays($expiresAt) . ')';
            }
            
            // إرسال رسالة النجاح
            $text = Replies::getForSuperDist('SUCCESS_ACTIVATION', $superdistId, [
                'account_name' => $actCode['account_name'],
                'account_slug' => $actCode['slug'],
                'access_type'  => $accessType === 'lifetime' ? 'مدى الحياة' : 'مؤقت',
                'expiry_info'  => $expiryInfo,
            ]);
            
            Telegram::sendMessage($chatId, $text);
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Failed to activate code', ['error' => $e->getMessage(), 'code' => $code]);
            Telegram::sendMessage($chatId, "❌ فشل تفعيل الكود. حاول مرة أخرى.");
        }
    }
    
    // ================================================================
    // دوال مساعدة
    // ================================================================
    
    /**
     * جلب معرف السوبر موزع للمستخدم
     * يستخدم لجلب الردود المخصصة
     */
    private static function getUserSuperDistId(int $userId): ?int
    {
        $superdistId = Db::fetchValue(
            "SELECT superdist_id FROM user_accounts WHERE user_id = ? AND superdist_id IS NOT NULL LIMIT 1",
            [$userId]
        );
        
        return $superdistId ? (int) $superdistId : null;
    }
    
    /**
     * إرسال رسالة خطأ الحد
     */
    private static function sendLimitError(int $chatId, array $limitCheck, ?int $superdistId): void
    {
        $reason = $limitCheck['reason'] ?? 'LIMIT_EXCEEDED';
        
        $replyKey = match ($reason) {
            'DAILY_LIMIT_EXCEEDED'   => 'ERROR_LIMIT_DAILY',
            'WEEKLY_LIMIT_EXCEEDED'  => 'ERROR_LIMIT_WEEKLY',
            'MONTHLY_LIMIT_EXCEEDED' => 'ERROR_LIMIT_MONTHLY',
            'BLOCKED'                => 'ERROR_USER_BANNED',
            default                  => 'ERROR_LIMIT_DAILY',
        };
        
        Telegram::sendMessage($chatId, Replies::getForSuperDist($replyKey, $superdistId, [
            'limit'      => $limitCheck['limit'] ?? 0,
            'used'       => $limitCheck['used'] ?? 0,
            'limit_type' => $limitCheck['limit_type'] ?? 'daily',
            'layer'      => $limitCheck['layer'] ?? 'global',
        ]));
    }
}
