<?php
/**
 * ============================================================
 * البوت العام - نقطة دخول Webhook
 * ============================================================
 * 
 * هذا الملف يستقبل التحديثات من Telegram ويوجّهها حسب نوع المستخدم:
 * - السوبر موزع → SuperDistributorCommands
 * - الزبون العادي → UserCommands
 * 
 * إعداد Webhook:
 *   https://api.telegram.org/bot{TOKEN}/setWebhook?url=https://your-domain.com/public_bot.php
 * 
 * ============================================================
 * تدفق العمل:
 * ============================================================
 * 1. استقبال التحديث من Telegram
 * 2. تحديد نوع المستخدم (سوبر موزع أو زبون)
 * 3. التحقق من الحظر
 * 4. التحقق من حالات الانتظار (State)
 * 5. توجيه الأمر للمعالج المناسب
 * 
 * ============================================================
 * فحص الاشتراك بالقناة:
 * ============================================================
 * يتم الفحص مركزياً عبر دالة requireChannelSubscription()
 * قبل أي عملية حساسة (تفعيل، طلب كود، توليد أكواد)
 */

declare(strict_types=1);

// تعريف الثوابت
define('BOT_STARTED', true);
define('BOT_TYPE', 'public');

// تحميل المكتبات
require __DIR__ . '/libs/bootstrap.php';

// تحميل معالجات الأوامر
require __DIR__ . '/src/public/SuperDistributorCommands.php';
require __DIR__ . '/src/public/UserCommands.php';

// ============================================================
// استقبال ومعالجة التحديث
// ============================================================

try {
    // قراءة التحديث من Telegram
    $input = file_get_contents('php://input');
    $update = json_decode($input, true);
    
    // التحقق من صحة التحديث
    if (!$update || !is_array($update)) {
        Logger::debug('Invalid update received', ['input' => substr($input, 0, 200)]);
        http_response_code(200);
        exit('OK');
    }
    
    Logger::debug('Public bot update received', ['update_id' => $update['update_id'] ?? 0]);
    
    // استخراج معلومات الرسالة
    $message = $update['message'] ?? $update['edited_message'] ?? null;
    $callbackQuery = $update['callback_query'] ?? null;
    
    // معالجة الرسائل
    if ($message) {
        handleMessage($message);
    }
    // معالجة Callback Query
    elseif ($callbackQuery) {
        handleCallbackQuery($callbackQuery);
    }
    
} catch (Throwable $e) {
    Logger::error('Public bot error', [
        'message' => $e->getMessage(),
        'file'    => $e->getFile(),
        'line'    => $e->getLine(),
    ]);
}

http_response_code(200);
exit('OK');

// ============================================================
// معالجة الرسائل
// ============================================================

/**
 * معالجة رسالة واردة
 */
function handleMessage(array $message): void
{
    $chatId = $message['chat']['id'] ?? 0;
    $fromId = $message['from']['id'] ?? 0;
    $text = trim($message['text'] ?? '');
    $username = $message['from']['username'] ?? null;
    $firstName = $message['from']['first_name'] ?? null;
    $lastName = $message['from']['last_name'] ?? null;
    
    // تحديد نوع المستخدم
    $userType = determineUserType($fromId);
    
    // ============================================================
    // معالجة السوبر موزع
    // ============================================================
    if ($userType['type'] === 'superdist') {
        $superdist = $userType['data'];
        
        // التحقق من الحظر/الإيقاف
        if ($superdist['status'] === 'banned') {
            Telegram::sendMessage($chatId, Replies::get('ERROR_SUPERDIST_BANNED'));
            return;
        }
        
        if ($superdist['status'] === 'suspended') {
            Telegram::sendMessage($chatId, Replies::get('ERROR_SUPERDIST_SUSPENDED'));
            return;
        }
        
        if ($superdist['status'] === 'archived') {
            Telegram::sendMessage($chatId, Replies::get('ERROR_SUPERDIST_ARCHIVED'));
            return;
        }
        
        // التحقق من حالات الانتظار
        $state = State::get($fromId);
        if ($state) {
            if (handleSuperDistStateInput($message, $state, $fromId, $chatId, $text, $superdist)) {
                return;
            }
        }
        
        // معالجة الأوامر والنصوص
        handleSuperDistMessage($text, $fromId, $chatId, $message, $superdist);
        return;
    }
    
    // ============================================================
    // معالجة الزبون العادي
    // ============================================================
    
    // إنشاء/جلب المستخدم
    $user = getOrCreateUser($fromId, $username, $firstName, $lastName);
    
    // التحقق من الحظر
    if (isUserBanned($user['id'])) {
        Telegram::sendMessage($chatId, Replies::get('ERROR_USER_BANNED'));
        return;
    }
    
    // التحقق من حالات الانتظار
    $state = State::get($fromId);
    if ($state) {
        if (handleUserStateInput($message, $state, $fromId, $chatId, $text, $user)) {
            return;
        }
    }
    
    // معالجة الأوامر والنصوص
    handleUserMessage($text, $fromId, $chatId, $message, $user);
}

/**
 * تحديد نوع المستخدم
 * 
 * @param int $telegramId
 * @return array ['type' => 'superdist'|'user', 'data' => array|null]
 */
function determineUserType(int $telegramId): array
{
    // البحث في السوبر موزعين أولاً
    $superdist = getSuperDistByTelegramId($telegramId);
    
    if ($superdist) {
        return [
            'type' => 'superdist',
            'data' => $superdist,
        ];
    }
    
    return [
        'type' => 'user',
        'data' => null,
    ];
}

/**
 * فحص الاشتراك بالقناة (مركزي)
 * 
 * @param int $chatId للرد
 * @param int $telegramId للفحص
 * @return bool true = مشترك، false = غير مشترك (تم إرسال رسالة الخطأ)
 */
function requireChannelSubscription(int $chatId, int $telegramId): bool
{
    if (checkChannelSubscription($telegramId)) {
        return true;
    }
    
    sendChannelSubscriptionError($chatId);
    return false;
}

// ============================================================
// معالجة رسائل السوبر موزع
// ============================================================

/**
 * معالجة رسالة السوبر موزع
 */
function handleSuperDistMessage(string $text, int $fromId, int $chatId, array $message, array $superdist): void
{
    // الأوامر الأساسية (لا تحتاج فحص قناة)
    if ($text === '/start') {
        SuperDistributorCommands::cmdStart($chatId, $superdist);
        return;
    }
    
    if ($text === '/help' || $text === '/superdist_help') {
        SuperDistributorCommands::cmdHelp($chatId);
        return;
    }
    
    if ($text === '/cancel' || $text === 'إلغاء') {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ تم الإلغاء.");
        return;
    }
    
    // معالجة الأوامر الأخرى
    if (strpos($text, '/') === 0 || strpos($text, 'gen_code') === 0 || strpos($text, 'gen_codes') === 0) {
        SuperDistributorCommands::handle($text, $fromId, $chatId, $message, $superdist);
        return;
    }
    
    // نص عادي - قد يكون كود اشتراك SD-... أو قسيمة VCH-...
    $upperText = strtoupper($text);
    
    // كود SD-... (اشتراك موزع قديم)
    if (strpos($upperText, 'SD-') === 0) {
        SuperDistributorCommands::handleSubscriptionCode($text, $fromId, $chatId, $superdist);
        return;
    }
    
    // كود VCH-... (قسيمة حسابات)
    if (strpos($upperText, 'VCH-') === 0) {
        SuperDistributorCommands::handleVoucherCode($text, $fromId, $chatId, $superdist);
        return;
    }
    
    // نص غير معروف
    Telegram::sendMessage($chatId, "❓ أمر غير معروف.\n\nاستخدم /help لعرض الأوامر.");
}

/**
 * معالجة حالات الانتظار للسوبر موزع
 */
function handleSuperDistStateInput(array $message, array $state, int $fromId, int $chatId, string $text, array $superdist): bool
{
    // إلغاء
    if (strtolower($text) === '/cancel' || $text === 'إلغاء') {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ تم الإلغاء.");
        return true;
    }
    
    switch ($state['state']) {
        case State::STATE_BROADCAST_AWAITING_CONTENT:
            handleSuperDistBroadcastContent($message, $fromId, $chatId, $superdist);
            return true;
            
        case State::STATE_BROADCAST_AWAITING_CONFIRM:
            handleSuperDistBroadcastConfirm($text, $fromId, $chatId, $superdist);
            return true;
    }
    
    return false;
}

/**
 * معالجة محتوى بث السوبر موزع
 */
function handleSuperDistBroadcastContent(array $message, int $fromId, int $chatId, array $superdist): void
{
    // التحقق من صلاحية البث
    if (!$superdist['can_broadcast']) {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ ليس لديك صلاحية البث.");
        return;
    }
    
    // تحديد نوع المحتوى
    $contentType = 'text';
    $contentText = null;
    $contentFileId = null;
    $caption = null;
    
    if (!empty($message['text'])) {
        $contentType = 'text';
        $contentText = $message['text'];
    } elseif (!empty($message['photo'])) {
        $contentType = 'photo';
        $photos = $message['photo'];
        $contentFileId = end($photos)['file_id'];
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['video'])) {
        $contentType = 'video';
        $contentFileId = $message['video']['file_id'];
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['document'])) {
        $contentType = 'document';
        $contentFileId = $message['document']['file_id'];
        $caption = $message['caption'] ?? '';
    } else {
        Telegram::sendMessage($chatId, "❌ نوع المحتوى غير مدعوم.");
        return;
    }
    
    // التحقق من نوع المحتوى المسموح
    $allowedType = $superdist['broadcast_content_type'];
    
    if ($allowedType === 'text_only' && $contentType !== 'text') {
        Telegram::sendMessage($chatId, "❌ مسموح لك بإرسال نص فقط.");
        return;
    }
    
    if ($allowedType === 'text_images' && !in_array($contentType, ['text', 'photo'])) {
        Telegram::sendMessage($chatId, "❌ مسموح لك بإرسال نص أو صور فقط.");
        return;
    }
    
    // حفظ المحتوى
    State::setBroadcastContent($fromId, $contentType, $contentText, $contentFileId, $caption);
    
    // حساب عدد المستلمين
    $recipientsCount = Db::count('user_accounts', 'superdist_id = ?', [$superdist['id']]);
    
    Telegram::sendMessage($chatId, Replies::get('INFO_BROADCAST_CONFIRM', [
        'content_type'     => $contentType,
        'recipients_count' => $recipientsCount,
    ]));
}

/**
 * معالجة تأكيد بث السوبر موزع
 */
function handleSuperDistBroadcastConfirm(string $text, int $fromId, int $chatId, array $superdist): void
{
    $text = mb_strtolower(trim($text));
    
    if ($text === 'تأكيد' || $text === 'confirm' || $text === 'نعم') {
        SuperDistributorCommands::executeBroadcast($fromId, $chatId, $superdist);
    } elseif ($text === 'تعديل' || $text === 'edit') {
        State::changeState($fromId, State::STATE_BROADCAST_AWAITING_CONTENT);
        Telegram::sendMessage($chatId, "📝 أرسل المحتوى الجديد.\n\nأو /cancel للإلغاء.");
    } else {
        Telegram::sendMessage($chatId, "⚠️ أرسل:\n• \"تأكيد\" للإرسال\n• \"تعديل\" لتغيير المحتوى\n• /cancel للإلغاء");
    }
}

// ============================================================
// معالجة رسائل الزبون
// ============================================================

/**
 * معالجة رسالة الزبون
 */
function handleUserMessage(string $text, int $fromId, int $chatId, array $message, array $user): void
{
    // الأوامر الأساسية (لا تحتاج فحص قناة)
    if ($text === '/start') {
        UserCommands::cmdStart($chatId, $user);
        return;
    }
    
    if ($text === '/help' || $text === '/user_help') {
        UserCommands::cmdHelp($chatId);
        return;
    }
    
    if ($text === '/cancel' || $text === 'إلغاء') {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ تم الإلغاء.");
        return;
    }
    
    // عرض حساباتي (لا تحتاج فحص قناة)
    if ($text === '/myaccounts' || $text === '/my_accounts') {
        UserCommands::cmdMyAccounts($chatId, $user);
        return;
    }
    
    // أوامر تحتاج فحص القناة
    if (strpos($text, '/') === 0 || strpos($text, 'get_code') === 0) {
        UserCommands::handle($text, $fromId, $chatId, $message, $user);
        return;
    }
    
    // نص عادي - قد يكون كود تفعيل ACT-...
    $upperText = strtoupper($text);
    if (strpos($upperText, 'ACT-') === 0) {
        // فحص الاشتراك بالقناة قبل التفعيل
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        UserCommands::handleActivationCode($text, $fromId, $chatId, $user);
        return;
    }
    
    // قد يكون اسم حساب (slug)
    $account = getAccountBySlug(strtolower($text));
    if ($account) {
        // فحص الاشتراك بالقناة قبل طلب الكود
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        UserCommands::handleGetCodeBySlug($text, $fromId, $chatId, $user);
        return;
    }
    
    // نص غير معروف
    Telegram::sendMessage($chatId, "❓ لم أفهم طلبك.\n\nاستخدم /help لعرض الأوامر المتاحة.\nأو أرسل كود التفعيل مباشرة.");
}

/**
 * معالجة حالات الانتظار للزبون
 */
function handleUserStateInput(array $message, array $state, int $fromId, int $chatId, string $text, array $user): bool
{
    // إلغاء
    if (strtolower($text) === '/cancel' || $text === 'إلغاء') {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ تم الإلغاء.");
        return true;
    }
    
    // حالياً لا توجد حالات انتظار للزبون
    return false;
}

// ============================================================
// معالجة Callback Query
// ============================================================

/**
 * معالجة Callback Query
 */
function handleCallbackQuery(array $callbackQuery): void
{
    $callbackId = $callbackQuery['id'];
    $fromId = $callbackQuery['from']['id'] ?? 0;
    $data = $callbackQuery['data'] ?? '';
    $chatId = $callbackQuery['message']['chat']['id'] ?? 0;
    
    // تحديد نوع المستخدم
    $userType = determineUserType($fromId);
    
    // معالجة حسب نوع المستخدم (للاستخدام المستقبلي)
    Telegram::answerCallbackQuery($callbackId);
}
