<?php
/**
 * نظام القسائم (Vouchers)
 * 
 * يدعم:
 * - إنشاء قسائم اشتراك موزعين (sd)
 * - إنشاء قسائم حزم أكواد تفعيل (activation_bundle)
 * - استرداد القسائم
 * - ربط الدفع
 */

class Voucher
{
    // أنواع القسائم
    const TYPE_SD = 'sd';
    const TYPE_ACTIVATION_BUNDLE = 'activation_bundle';
    
    // حالات القسيمة
    const STATUS_UNUSED = 'unused';
    const STATUS_PARTIAL = 'partial';
    const STATUS_USED = 'used';
    const STATUS_EXPIRED = 'expired';
    const STATUS_DISABLED = 'disabled';
    
    /**
     * إنشاء قسيمة جديدة
     * 
     * @param string $type نوع القسيمة (sd, activation_bundle)
     * @param array $payload بيانات القسيمة
     * @param array $options خيارات إضافية
     * @return array ['success' => bool, 'code' => string, 'voucher_id' => int, 'error' => string]
     */
    public static function create(string $type, array $payload, array $options = []): array
    {
        // التحقق من النوع
        if (!in_array($type, [self::TYPE_SD, self::TYPE_ACTIVATION_BUNDLE])) {
            return ['success' => false, 'error' => 'نوع القسيمة غير صالح'];
        }
        
        // التحقق من payload
        $validationResult = self::validatePayload($type, $payload);
        if (!$validationResult['valid']) {
            return ['success' => false, 'error' => $validationResult['error']];
        }
        
        // توليد كود فريد
        $code = self::generateCode();
        
        // الخيارات الافتراضية
        $maxUses = $options['max_uses'] ?? 1;
        $expiresAt = $options['expires_at'] ?? null;
        $createdByAdminId = $options['created_by_admin_id'] ?? null;
        $paymentId = $options['payment_id'] ?? null;
        $paymentProvider = $options['payment_provider'] ?? null;
        $paymentAmount = $options['payment_amount'] ?? null;
        $paymentCurrency = $options['payment_currency'] ?? null;
        
        try {
            Db::insert('vouchers', [
                'code' => $code,
                'type' => $type,
                'payload' => json_encode($payload, JSON_UNESCAPED_UNICODE),
                'max_uses' => $maxUses,
                'current_uses' => 0,
                'status' => self::STATUS_UNUSED,
                'expires_at' => $expiresAt,
                'payment_id' => $paymentId,
                'payment_provider' => $paymentProvider,
                'payment_amount' => $paymentAmount,
                'payment_currency' => $paymentCurrency,
                'created_by_admin_id' => $createdByAdminId,
            ]);
            
            $voucherId = Db::lastInsertId();
            
            Logger::event('voucher_created', 'voucher', $voucherId, 'admin', $createdByAdminId, [
                'code' => $code,
                'type' => $type,
                'payload' => $payload,
            ]);
            
            return [
                'success' => true,
                'code' => $code,
                'voucher_id' => $voucherId,
            ];
            
        } catch (Throwable $e) {
            Logger::error('Failed to create voucher', ['error' => $e->getMessage()]);
            return ['success' => false, 'error' => 'فشل إنشاء القسيمة'];
        }
    }
    
    /**
     * التحقق من صحة payload
     */
    private static function validatePayload(string $type, array $payload): array
    {
        if ($type === self::TYPE_SD) {
            if (empty($payload['accounts']) || !is_array($payload['accounts'])) {
                return ['valid' => false, 'error' => 'يجب تحديد الحسابات'];
            }
            foreach ($payload['accounts'] as $acc) {
                if (empty($acc['account_id']) || empty($acc['duration_days'])) {
                    return ['valid' => false, 'error' => 'بيانات الحساب ناقصة'];
                }
            }
        } elseif ($type === self::TYPE_ACTIVATION_BUNDLE) {
            if (empty($payload['account_id'])) {
                return ['valid' => false, 'error' => 'يجب تحديد الحساب'];
            }
            if (empty($payload['codes_count']) || $payload['codes_count'] < 1) {
                return ['valid' => false, 'error' => 'عدد الأكواد غير صالح'];
            }
        }
        
        return ['valid' => true];
    }
    
    /**
     * توليد كود قسيمة فريد
     */
    public static function generateCode(): string
    {
        $prefix = 'VCH';
        $part1 = strtoupper(substr(bin2hex(random_bytes(2)), 0, 4));
        $part2 = strtoupper(substr(bin2hex(random_bytes(2)), 0, 4));
        return "{$prefix}-{$part1}-{$part2}";
    }
    
    /**
     * جلب قسيمة بالكود
     */
    public static function getByCode(string $code): ?array
    {
        return Db::fetchOne(
            "SELECT * FROM vouchers WHERE code = ?",
            [strtoupper($code)]
        );
    }
    
    /**
     * جلب قسيمة بالمعرف
     */
    public static function getById(int $id): ?array
    {
        return Db::fetchOne(
            "SELECT * FROM vouchers WHERE id = ?",
            [$id]
        );
    }
    
    /**
     * استرداد قسيمة
     * 
     * @param string $code كود القسيمة
     * @param int $telegramId Telegram ID للمستفيد
     * @return array ['success' => bool, 'result' => array, 'error' => string]
     */
    public static function redeem(string $code, int $telegramId): array
    {
        $code = strtoupper(trim($code));
        
        try {
            Db::beginTransaction();
            
            // جلب القسيمة مع قفل
            $voucher = Db::fetchOne(
                "SELECT * FROM vouchers WHERE code = ? FOR UPDATE",
                [$code]
            );
            
            if (!$voucher) {
                Db::rollback();
                return ['success' => false, 'error' => 'القسيمة غير موجودة'];
            }
            
            // التحقق من الحالة
            if ($voucher['status'] === self::STATUS_USED) {
                Db::rollback();
                return ['success' => false, 'error' => 'القسيمة مستخدمة بالكامل'];
            }
            
            if ($voucher['status'] === self::STATUS_EXPIRED) {
                Db::rollback();
                return ['success' => false, 'error' => 'القسيمة منتهية الصلاحية'];
            }
            
            if ($voucher['status'] === self::STATUS_DISABLED) {
                Db::rollback();
                return ['success' => false, 'error' => 'القسيمة معطلة'];
            }
            
            // التحقق من تاريخ الانتهاء
            if ($voucher['expires_at'] && strtotime($voucher['expires_at']) < time()) {
                Db::update('vouchers', ['status' => self::STATUS_EXPIRED], 'id = ?', [$voucher['id']]);
                Db::commit();
                return ['success' => false, 'error' => 'القسيمة منتهية الصلاحية'];
            }
            
            // التحقق من عدد الاستخدامات
            if ($voucher['current_uses'] >= $voucher['max_uses']) {
                Db::update('vouchers', ['status' => self::STATUS_USED], 'id = ?', [$voucher['id']]);
                Db::commit();
                return ['success' => false, 'error' => 'القسيمة مستخدمة بالكامل'];
            }
            
            // تنفيذ الاسترداد حسب النوع
            $payload = json_decode($voucher['payload'], true);
            $result = [];
            
            if ($voucher['type'] === self::TYPE_SD) {
                $result = self::redeemSD($voucher, $payload, $telegramId);
            } elseif ($voucher['type'] === self::TYPE_ACTIVATION_BUNDLE) {
                $result = self::redeemActivationBundle($voucher, $payload, $telegramId);
            }
            
            if (!$result['success']) {
                Db::rollback();
                return $result;
            }
            
            // تحديث القسيمة
            $newUses = $voucher['current_uses'] + 1;
            $newStatus = ($newUses >= $voucher['max_uses']) ? self::STATUS_USED : self::STATUS_PARTIAL;
            
            Db::update('vouchers', [
                'current_uses' => $newUses,
                'status' => $newStatus,
            ], 'id = ?', [$voucher['id']]);
            
            // تسجيل الاسترداد
            Db::insert('voucher_redemptions', [
                'voucher_id' => $voucher['id'],
                'redeemer_telegram_id' => $telegramId,
                'redeemer_type' => $result['redeemer_type'],
                'redeemer_id' => $result['redeemer_id'] ?? null,
                'result' => json_encode($result['data'], JSON_UNESCAPED_UNICODE),
            ]);
            
            Db::commit();
            
            Logger::event('voucher_redeemed', 'voucher', $voucher['id'], 'telegram', $telegramId, [
                'type' => $voucher['type'],
                'result' => $result['data'],
            ]);
            
            return [
                'success' => true,
                'result' => $result['data'],
                'voucher_type' => $voucher['type'],
            ];
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Voucher redeem failed', ['code' => $code, 'error' => $e->getMessage()]);
            return ['success' => false, 'error' => 'فشل استرداد القسيمة'];
        }
    }
    
    /**
     * استرداد قسيمة نوع SD (اشتراك موزع)
     */
    private static function redeemSD(array $voucher, array $payload, int $telegramId): array
    {
        // جلب أو إنشاء الموزع
        $superdist = getSuperDistByTelegramId($telegramId);
        
        if (!$superdist) {
            // إنشاء موزع جديد
            Db::insert('super_distributors', [
                'telegram_id' => $telegramId,
                'status' => 'active',
                'can_broadcast' => 1,
                'broadcast_allowed_types' => '["text","stickers","photos","videos","documents","audio"]',
            ]);
            $superdistId = Db::lastInsertId();
            $superdist = ['id' => $superdistId, 'telegram_id' => $telegramId];
        } else {
            $superdistId = $superdist['id'];
        }
        
        $addedAccounts = [];
        
        foreach ($payload['accounts'] as $accData) {
            $accountId = $accData['account_id'];
            $durationDays = $accData['duration_days'];
            $limits = $accData['limits'] ?? [];
            
            // التحقق من وجود الحساب
            $account = Db::fetchOne("SELECT * FROM accounts WHERE id = ?", [$accountId]);
            if (!$account) {
                continue;
            }
            
            // التحقق من وجود ربط سابق
            $existing = Db::fetchOne(
                "SELECT * FROM superdist_accounts WHERE superdist_id = ? AND account_id = ?",
                [$superdistId, $accountId]
            );
            
            $startAt = date('Y-m-d H:i:s');
            $endAt = date('Y-m-d H:i:s', strtotime("+{$durationDays} days"));
            
            if ($existing) {
                // تمديد الصلاحية
                $currentEnd = strtotime($existing['end_at']);
                if ($currentEnd > time()) {
                    // تمديد من النهاية الحالية
                    $endAt = date('Y-m-d H:i:s', $currentEnd + ($durationDays * 86400));
                }
                
                Db::update('superdist_accounts', [
                    'end_at' => $endAt,
                    'activation_limit_daily' => $limits['daily'] ?? $existing['activation_limit_daily'],
                    'activation_limit_weekly' => $limits['weekly'] ?? $existing['activation_limit_weekly'],
                    'activation_limit_monthly' => $limits['monthly'] ?? $existing['activation_limit_monthly'],
                ], 'id = ?', [$existing['id']]);
            } else {
                // إنشاء ربط جديد
                Db::insert('superdist_accounts', [
                    'superdist_id' => $superdistId,
                    'account_id' => $accountId,
                    'start_at' => $startAt,
                    'duration_days' => $durationDays,
                    'end_at' => $endAt,
                    'activation_limit_daily' => $limits['daily'] ?? null,
                    'activation_limit_weekly' => $limits['weekly'] ?? null,
                    'activation_limit_monthly' => $limits['monthly'] ?? null,
                ]);
            }
            
            $addedAccounts[] = [
                'account_id' => $accountId,
                'account_name' => $account['name'],
                'duration_days' => $durationDays,
                'end_at' => $endAt,
            ];
        }
        
        return [
            'success' => true,
            'redeemer_type' => 'superdist',
            'redeemer_id' => $superdistId,
            'data' => [
                'superdist_id' => $superdistId,
                'accounts_added' => $addedAccounts,
            ],
        ];
    }
    
    /**
     * استرداد قسيمة نوع activation_bundle (حزمة أكواد)
     */
    private static function redeemActivationBundle(array $voucher, array $payload, int $telegramId): array
    {
        // يجب أن يكون المستفيد موزع
        $superdist = getSuperDistByTelegramId($telegramId);
        
        if (!$superdist) {
            return ['success' => false, 'error' => 'يجب أن تكون موزعاً لاستخدام هذه القسيمة'];
        }
        
        $accountId = $payload['account_id'];
        $codesCount = $payload['codes_count'];
        $accessType = $payload['access_type'] ?? 'lifetime';
        $accessDays = $payload['access_days'] ?? null;
        
        // التحقق من الحساب
        $account = Db::fetchOne("SELECT * FROM accounts WHERE id = ?", [$accountId]);
        if (!$account) {
            return ['success' => false, 'error' => 'الحساب غير موجود'];
        }
        
        // توليد الأكواد
        $generatedCodes = [];
        
        for ($i = 0; $i < $codesCount; $i++) {
            $code = CodeGenerator::generateActivationCode();
            
            Db::insert('activation_codes', [
                'code' => $code,
                'account_id' => $accountId,
                'superdist_id' => $superdist['id'],
                'access_type' => $accessType,
                'access_days' => $accessDays,
                'status' => 'unused',
            ]);
            
            $codeId = Db::lastInsertId();
            
            // ربط الكود بالقسيمة
            Db::insert('voucher_codes', [
                'voucher_id' => $voucher['id'],
                'activation_code_id' => $codeId,
            ]);
            
            $generatedCodes[] = $code;
        }
        
        return [
            'success' => true,
            'redeemer_type' => 'superdist',
            'redeemer_id' => $superdist['id'],
            'data' => [
                'account_id' => $accountId,
                'account_name' => $account['name'],
                'codes_count' => $codesCount,
                'codes' => $generatedCodes,
                'access_type' => $accessType,
                'access_days' => $accessDays,
            ],
        ];
    }
    
    /**
     * تعطيل قسيمة
     */
    public static function disable(int $id): bool
    {
        return Db::update('vouchers', ['status' => self::STATUS_DISABLED], 'id = ?', [$id]) > 0;
    }
    
    /**
     * جلب قسائم بفلاتر
     */
    public static function getList(array $filters = [], int $limit = 50, int $offset = 0): array
    {
        $where = ['1=1'];
        $params = [];
        
        if (!empty($filters['type'])) {
            $where[] = 'type = ?';
            $params[] = $filters['type'];
        }
        
        if (!empty($filters['status'])) {
            $where[] = 'status = ?';
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['created_by_admin_id'])) {
            $where[] = 'created_by_admin_id = ?';
            $params[] = $filters['created_by_admin_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        $params[] = $limit;
        $params[] = $offset;
        
        return Db::fetchAll(
            "SELECT * FROM vouchers WHERE {$whereClause} ORDER BY created_at DESC LIMIT ? OFFSET ?",
            $params
        );
    }
    
    /**
     * إنشاء قسيمة من webhook الدفع
     */
    public static function createFromPayment(array $paymentData): array
    {
        // استخراج البيانات من payload الدفع
        $paymentId = $paymentData['payment_id'];
        $provider = $paymentData['provider'];
        $amount = $paymentData['amount'];
        $currency = $paymentData['currency'];
        $metadata = $paymentData['metadata'] ?? [];
        
        // تحديد نوع القسيمة من metadata
        $type = $metadata['voucher_type'] ?? self::TYPE_SD;
        $payload = $metadata['payload'] ?? [];
        
        return self::create($type, $payload, [
            'payment_id' => $paymentId,
            'payment_provider' => $provider,
            'payment_amount' => $amount,
            'payment_currency' => $currency,
            'payment_status' => 'completed',
        ]);
    }
}
