<?php
/**
 * ============================================================
 * مكتبة إدارة حالات الجلسات (State Manager)
 * ============================================================
 * 
 * تدير الحالات المؤقتة للمستخدمين، مثل:
 * - انتظار محتوى البث
 * - انتظار تأكيد البث
 * - انتظار إدخال قيمة
 * 
 * الاستخدام:
 *   State::set($telegramId, 'admin', 'awaiting_broadcast_content', ['audience' => 'all']);
 *   $state = State::get($telegramId);
 *   State::clear($telegramId);
 */

class State
{
    /** @var int مدة انتهاء الحالة الافتراضية (ساعة واحدة) */
    private const DEFAULT_EXPIRY_SECONDS = 3600;
    
    /**
     * الحالات المعرّفة
     */
    public const STATE_NONE = 'none';
    
    // حالات البث
    public const STATE_BROADCAST_AWAITING_CONTENT = 'awaiting_broadcast_content';
    public const STATE_BROADCAST_AWAITING_CONFIRM = 'awaiting_broadcast_confirm';
    
    // حالات الإدارة - الردود
    public const STATE_AWAITING_REPLY_TEXT = 'awaiting_reply_text';
    public const STATE_AWAITING_SUPERDIST_REPLY_TEXT = 'awaiting_superdist_reply_text';
    
    // حالات الإدارة - عام
    public const STATE_AWAITING_INPUT = 'awaiting_input';
    
    /**
     * تعيين حالة للمستخدم
     * 
     * @param int $telegramId معرف تيليجرام
     * @param string $userType نوع المستخدم (admin, superdist, user)
     * @param string $state الحالة
     * @param array|null $data بيانات الحالة
     * @param int|null $expirySeconds مدة الانتهاء بالثواني
     */
    public static function set(
        int $telegramId,
        string $userType,
        string $state,
        ?array $data = null,
        ?int $expirySeconds = null
    ): void {
        $expiresAt = date('Y-m-d H:i:s', time() + ($expirySeconds ?? self::DEFAULT_EXPIRY_SECONDS));
        
        // حذف الحالة القديمة إذا وجدت
        self::clear($telegramId);
        
        Db::insert('user_states', [
            'telegram_id' => $telegramId,
            'user_type'   => $userType,
            'state'       => $state,
            'state_data'  => $data !== null ? json_encode($data, JSON_UNESCAPED_UNICODE) : null,
            'expires_at'  => $expiresAt,
        ]);
    }
    
    /**
     * جلب حالة المستخدم
     * 
     * @param int $telegramId
     * @return array|null ['state' => string, 'data' => array|null, 'user_type' => string]
     */
    public static function get(int $telegramId): ?array
    {
        $row = Db::fetchOne(
            "SELECT * FROM user_states WHERE telegram_id = ? AND (expires_at IS NULL OR expires_at > NOW())",
            [$telegramId]
        );
        
        if (!$row) {
            return null;
        }
        
        return [
            'state'     => $row['state'],
            'data'      => $row['state_data'] ? json_decode($row['state_data'], true) : null,
            'user_type' => $row['user_type'],
            'created_at'=> $row['created_at'],
            'expires_at'=> $row['expires_at'],
        ];
    }
    
    /**
     * التحقق من وجود حالة محددة
     * 
     * @param int $telegramId
     * @param string $state الحالة للتحقق منها
     * @return bool
     */
    public static function is(int $telegramId, string $state): bool
    {
        $current = self::get($telegramId);
        return $current !== null && $current['state'] === $state;
    }
    
    /**
     * التحقق من وجود أي حالة نشطة
     * 
     * @param int $telegramId
     * @return bool
     */
    public static function hasActiveState(int $telegramId): bool
    {
        return self::get($telegramId) !== null;
    }
    
    /**
     * تحديث بيانات الحالة
     * 
     * @param int $telegramId
     * @param array $data البيانات الجديدة (تُدمج مع القديمة)
     */
    public static function updateData(int $telegramId, array $data): void
    {
        $current = self::get($telegramId);
        
        if (!$current) {
            return;
        }
        
        $newData = array_merge($current['data'] ?? [], $data);
        
        Db::update('user_states', [
            'state_data' => json_encode($newData, JSON_UNESCAPED_UNICODE),
        ], 'telegram_id = ?', [$telegramId]);
    }
    
    /**
     * تغيير الحالة مع الحفاظ على البيانات
     * 
     * @param int $telegramId
     * @param string $newState
     */
    public static function changeState(int $telegramId, string $newState): void
    {
        Db::update('user_states', [
            'state' => $newState,
        ], 'telegram_id = ?', [$telegramId]);
    }
    
    /**
     * مسح حالة المستخدم
     * 
     * @param int $telegramId
     */
    public static function clear(int $telegramId): void
    {
        Db::delete('user_states', 'telegram_id = ?', [$telegramId]);
    }
    
    /**
     * تنظيف الحالات المنتهية
     * 
     * يُستدعى دورياً لحذف الحالات القديمة
     * 
     * @return int عدد الحالات المحذوفة
     */
    public static function cleanupExpired(): int
    {
        return Db::execute("DELETE FROM user_states WHERE expires_at IS NOT NULL AND expires_at < NOW()");
    }
    
    // ================================================================
    // دوال مساعدة للبث
    // ================================================================
    
    /**
     * بدء حالة انتظار محتوى البث
     * 
     * @param int $telegramId
     * @param string $userType
     * @param string $audienceType نوع الجمهور (all, users, superdists, superdist_users, account_users)
     * @param string|null $audienceFilter فلتر إضافي
     */
    public static function startBroadcast(
        int $telegramId,
        string $userType,
        string $audienceType,
        ?string $audienceFilter = null
    ): void {
        self::set($telegramId, $userType, self::STATE_BROADCAST_AWAITING_CONTENT, [
            'audience_type'   => $audienceType,
            'audience_filter' => $audienceFilter,
        ]);
    }
    
    /**
     * تخزين محتوى البث والانتقال لحالة التأكيد
     * 
     * @param int $telegramId
     * @param string $contentType نوع المحتوى
     * @param string|null $contentText النص
     * @param string|null $contentFileId معرف الملف
     * @param string|null $caption الوصف
     */
    public static function setBroadcastContent(
        int $telegramId,
        string $contentType,
        ?string $contentText = null,
        ?string $contentFileId = null,
        ?string $caption = null
    ): void {
        self::updateData($telegramId, [
            'content_type'    => $contentType,
            'content_text'    => $contentText,
            'content_file_id' => $contentFileId,
            'content_caption' => $caption,
        ]);
        
        self::changeState($telegramId, self::STATE_BROADCAST_AWAITING_CONFIRM);
    }
    
    /**
     * جلب بيانات البث المعلّق
     * 
     * @param int $telegramId
     * @return array|null
     */
    public static function getBroadcastData(int $telegramId): ?array
    {
        $state = self::get($telegramId);
        
        if (!$state || !in_array($state['state'], [
            self::STATE_BROADCAST_AWAITING_CONTENT,
            self::STATE_BROADCAST_AWAITING_CONFIRM,
        ])) {
            return null;
        }
        
        return $state['data'];
    }
    
    // ================================================================
    // دوال مساعدة للإدخال
    // ================================================================
    
    /**
     * بدء انتظار إدخال
     * 
     * @param int $telegramId
     * @param string $userType
     * @param string $inputType نوع الإدخال المطلوب
     * @param array $context سياق إضافي
     */
    public static function awaitInput(
        int $telegramId,
        string $userType,
        string $inputType,
        array $context = []
    ): void {
        self::set($telegramId, $userType, self::STATE_AWAITING_INPUT, array_merge([
            'input_type' => $inputType,
        ], $context));
    }
    
    /**
     * التحقق من انتظار إدخال محدد
     * 
     * @param int $telegramId
     * @param string $inputType
     * @return bool
     */
    public static function isAwaitingInput(int $telegramId, string $inputType): bool
    {
        $state = self::get($telegramId);
        
        if (!$state || $state['state'] !== self::STATE_AWAITING_INPUT) {
            return false;
        }
        
        return ($state['data']['input_type'] ?? null) === $inputType;
    }
}
