<?php
/**
 * ============================================================
 * مكتبة الردود (Replies System)
 * ============================================================
 * 
 * تدير نصوص الرسائل مع دعم:
 * - الردود الافتراضية (من جدول replies)
 * - الردود المخصصة لسوبر موزع (من جدول superdist_replies)
 * - استبدال المتغيرات (placeholders)
 * 
 * ============================================================
 * المنطق الأساسي:
 * ============================================================
 * 
 * Replies::get($key, $placeholders):
 *   → يرجع دائماً من جدول replies (الرد العام/الافتراضي)
 *   → لا يأخذ السوبر موزع بالاعتبار
 * 
 * Replies::getForSuperDist($key, $superdistId, $placeholders):
 *   → يحاول أولاً قراءة الرد من superdist_replies للسوبر موزع المحدد
 *   → إذا لم يوجد رد مخصص، يرجع default_text من جدول replies
 *   → هذا يسمح بتخصيص رسائل معينة لزبائن سوبر موزع محدد
 * 
 * ============================================================
 * 
 * الاستخدام:
 *   $text = Replies::get('WELCOME_USER', ['name' => 'أحمد']);
 *   $text = Replies::getForSuperDist('ERROR_LIMIT_DAILY', 5, ['limit' => 3]);
 */

class Replies
{
    /** @var array كاش للردود الافتراضية */
    private static array $cache = [];
    
    /** @var bool هل تم تحميل الكاش */
    private static bool $cacheLoaded = false;
    
    /**
     * جلب رد افتراضي
     * 
     * @param string $key مفتاح الرد
     * @param array $placeholders المتغيرات للاستبدال
     * @return string النص النهائي
     */
    public static function get(string $key, array $placeholders = []): string
    {
        self::loadCache();
        
        $text = self::$cache[$key] ?? self::getFromDb($key);
        
        if ($text === null) {
            // رد افتراضي إذا لم يوجد
            return "[{$key}]";
        }
        
        return self::replacePlaceholders($text, $placeholders);
    }
    
    /**
     * جلب رد مع مراعاة السوبر موزع
     * 
     * يبحث أولاً عن رد مخصص للسوبر موزع، إذا لم يجد يستخدم الافتراضي
     * 
     * @param string $key مفتاح الرد
     * @param int|null $superdistId معرف السوبر موزع
     * @param array $placeholders المتغيرات للاستبدال
     * @return string
     */
    public static function getForSuperDist(string $key, ?int $superdistId, array $placeholders = []): string
    {
        // إذا لا يوجد سوبر موزع، استخدم الافتراضي
        if ($superdistId === null) {
            return self::get($key, $placeholders);
        }
        
        // البحث عن رد مخصص
        $customText = Db::fetchValue(
            "SELECT custom_text FROM superdist_replies WHERE superdist_id = ? AND reply_key = ?",
            [$superdistId, $key]
        );
        
        if ($customText) {
            return self::replacePlaceholders($customText, $placeholders);
        }
        
        // لم يوجد مخصص، استخدم الافتراضي
        return self::get($key, $placeholders);
    }
    
    /**
     * استبدال المتغيرات في النص
     * 
     * يدعم صيغتين:
     * - {variable}
     * - :variable
     * 
     * @param string $text النص
     * @param array $placeholders المتغيرات [key => value]
     * @return string
     */
    public static function replacePlaceholders(string $text, array $placeholders): string
    {
        foreach ($placeholders as $key => $value) {
            // صيغة {variable}
            $text = str_replace('{' . $key . '}', (string) $value, $text);
            // صيغة :variable
            $text = str_replace(':' . $key, (string) $value, $text);
        }
        
        return $text;
    }
    
    /**
     * تحميل جميع الردود الافتراضية إلى الكاش
     */
    private static function loadCache(): void
    {
        if (self::$cacheLoaded) {
            return;
        }
        
        $rows = Db::fetchAll("SELECT `key`, default_text FROM replies");
        
        foreach ($rows as $row) {
            self::$cache[$row['key']] = $row['default_text'];
        }
        
        self::$cacheLoaded = true;
    }
    
    /**
     * جلب رد من قاعدة البيانات مباشرة
     * 
     * @param string $key
     * @return string|null
     */
    private static function getFromDb(string $key): ?string
    {
        $text = Db::fetchValue(
            "SELECT default_text FROM replies WHERE `key` = ?",
            [$key]
        );
        
        if ($text) {
            self::$cache[$key] = $text;
        }
        
        return $text ?: null;
    }
    
    /**
     * مسح الكاش (بعد تحديث الردود)
     */
    public static function clearCache(): void
    {
        self::$cache = [];
        self::$cacheLoaded = false;
    }
    
    // ================================================================
    // دوال الإدارة
    // ================================================================
    
    /**
     * تعيين رد افتراضي
     * 
     * @param string $key المفتاح
     * @param string $text النص
     * @param string|null $description الوصف
     * @param string|null $placeholders المتغيرات المتاحة
     */
    public static function setDefault(string $key, string $text, ?string $description = null, ?string $placeholders = null): void
    {
        $exists = Db::exists('replies', '`key` = ?', [$key]);
        
        if ($exists) {
            Db::update('replies', [
                'default_text' => $text,
                'description'  => $description,
                'placeholders' => $placeholders,
            ], '`key` = ?', [$key]);
        } else {
            Db::insert('replies', [
                'key'          => $key,
                'default_text' => $text,
                'description'  => $description,
                'placeholders' => $placeholders,
            ]);
        }
        
        self::clearCache();
    }
    
    /**
     * تعيين رد مخصص لسوبر موزع
     * 
     * @param int $superdistId
     * @param string $key
     * @param string $text
     */
    public static function setSuperDistReply(int $superdistId, string $key, string $text): void
    {
        $exists = Db::exists(
            'superdist_replies',
            'superdist_id = ? AND reply_key = ?',
            [$superdistId, $key]
        );
        
        if ($exists) {
            Db::update('superdist_replies', [
                'custom_text' => $text,
            ], 'superdist_id = ? AND reply_key = ?', [$superdistId, $key]);
        } else {
            Db::insert('superdist_replies', [
                'superdist_id' => $superdistId,
                'reply_key'    => $key,
                'custom_text'  => $text,
            ]);
        }
    }
    
    /**
     * حذف رد مخصص لسوبر موزع
     * 
     * @param int $superdistId
     * @param string $key
     */
    public static function deleteSuperDistReply(int $superdistId, string $key): void
    {
        Db::delete('superdist_replies', 'superdist_id = ? AND reply_key = ?', [$superdistId, $key]);
    }
    
    /**
     * جلب قائمة مفاتيح الردود
     * 
     * @return array
     */
    public static function listKeys(): array
    {
        return Db::fetchAll("SELECT `key`, description, placeholders FROM replies ORDER BY `key`");
    }
    
    /**
     * جلب ردود سوبر موزع المخصصة
     * 
     * @param int $superdistId
     * @return array
     */
    public static function getSuperDistReplies(int $superdistId): array
    {
        return Db::fetchAll(
            "SELECT reply_key, custom_text FROM superdist_replies WHERE superdist_id = ?",
            [$superdistId]
        );
    }
    
    /**
     * اختبار رد (جلب النص النهائي)
     * 
     * @param string $key
     * @param int|null $superdistId
     * @return array ['key' => string, 'text' => string, 'is_custom' => bool]
     */
    public static function testReply(string $key, ?int $superdistId = null): array
    {
        $isCustom = false;
        
        if ($superdistId !== null) {
            $customText = Db::fetchValue(
                "SELECT custom_text FROM superdist_replies WHERE superdist_id = ? AND reply_key = ?",
                [$superdistId, $key]
            );
            
            if ($customText) {
                $isCustom = true;
                $text = $customText;
            }
        }
        
        if (!isset($text)) {
            self::loadCache();
            $text = self::$cache[$key] ?? self::getFromDb($key) ?? "[{$key}] غير موجود";
        }
        
        return [
            'key'       => $key,
            'text'      => $text,
            'is_custom' => $isCustom,
        ];
    }
}
