<?php
/**
 * أوامر إدارة صلاحيات الوصول الدفعي
 * يتم استدعاؤها من AdminCommands
 */

class BatchAccessCommands
{
    /**
     * تغيير صلاحيات وصول الزبائن دفعة واحدة
     * /set_users_access <accounts|all> <timed|lifetime> [days] [--preview] [--force] [--notify]
     */
    public static function cmdSetUsersAccess(int $chatId, string $args, int $fromId): void
    {
        try {
            // تحليل الأرجيومنتات
            $parts = preg_split('/\s+/', trim($args));
            
            if (count($parts) < 2) {
                self::showSetUsersAccessHelp($chatId);
                return;
            }
            
            $accountsArg = $parts[0];
            $newAccessType = strtolower($parts[1]);
            $newAccessDays = null;
            $isPreview = in_array('--preview', $parts);
            $force = in_array('--force', $parts);
            $notify = in_array('--notify', $parts);
            
            // التحقق من نوع الوصول
            if (!in_array($newAccessType, ['lifetime', 'timed'])) {
                Telegram::sendMessage($chatId, "❌ نوع الوصول يجب أن يكون: <code>lifetime</code> أو <code>timed</code>");
                return;
            }
            
            // إذا كان timed، يجب تحديد الأيام
            if ($newAccessType === 'timed') {
                foreach ($parts as $part) {
                    if (is_numeric($part) && $part > 0) {
                        $newAccessDays = (int) $part;
                        break;
                    }
                }
                
                if ($newAccessDays === null) {
                    Telegram::sendMessage($chatId, "❌ عند اختيار <code>timed</code>، يجب تحديد عدد الأيام\n\nمثال:\n<code>/set_users_access vip-acc timed 30</code>");
                    return;
                }
            }
            
            // تحديد الحسابات
            Telegram::sendMessage($chatId, "⏳ جاري البحث عن الحسابات...");
            
            $accountIds = [];
            if (strtolower($accountsArg) === 'all') {
                // جميع الحسابات
                $accountIds = Db::fetchColumn("SELECT id FROM accounts WHERE status = 'active'");
            } else {
                // حسابات محددة (IDs أو slugs)
                $accountsInput = explode(',', $accountsArg);
                foreach ($accountsInput as $input) {
                    $input = trim($input);
                    if (is_numeric($input)) {
                        $accountIds[] = (int) $input;
                    } else {
                        // slug
                        $account = Db::fetchOne("SELECT id FROM accounts WHERE slug = ?", [$input]);
                        if ($account) {
                            $accountIds[] = $account['id'];
                        }
                    }
                }
            }
            
            if (empty($accountIds)) {
                Telegram::sendMessage($chatId, "❌ لم يتم العثور على حسابات صالحة\n\nتأكد من وجود حسابات نشطة (status = active)");
                return;
            }
            
            Telegram::sendMessage($chatId, "✅ تم العثور على " . count($accountIds) . " حساب(ات)\n\n🔄 جاري " . ($isPreview ? "المعاينة" : "التنفيذ") . "...");
            
            // معاينة أو تنفيذ
            if ($isPreview) {
                self::executePreview($chatId, $accountIds, $newAccessType, $newAccessDays, $force);
            } else {
                self::executeChange($chatId, $accountIds, $newAccessType, $newAccessDays, $force, $notify, $fromId);
            }
            
        } catch (Throwable $e) {
            Telegram::sendMessage($chatId, "❌ خطأ غير متوقع:\n<code>" . htmlspecialchars($e->getMessage()) . "</code>\n\nالملف: " . basename($e->getFile()) . "\nالسطر: " . $e->getLine());
        }
    }
    
    /**
     * تنفيذ المعاينة
     */
    private static function executePreview(int $chatId, array $accountIds, string $newAccessType, ?int $newAccessDays, bool $force): void
    {
        try {
            $preview = BatchAccessManager::previewUsersAccessChange($accountIds, $newAccessType, $newAccessDays, $force);
        } catch (Throwable $e) {
            Telegram::sendMessage($chatId, "❌ خطأ في المعاينة:\n<code>" . htmlspecialchars($e->getMessage()) . "</code>");
            return;
        }
        
        $stats = $preview['stats'];
        $samples = $preview['samples'];
        
        $text = "📋 <b>معاينة تغيير صلاحيات الوصول</b>\n";
        $text .= "⚠️ <b>المعاينة فقط — لم يتم تطبيق أي تغيير</b>\n\n";
        
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "📊 <b>الإحصائيات:</b>\n";
        $text .= "• إجمالي السجلات المتأثرة: {$stats['total_affected']}\n";
        $text .= "• عدد الزبائن الفريدين: {$stats['unique_users']}\n";
        $text .= "• عدد الحسابات: {$stats['unique_accounts']}\n\n";
        
        $text .= "📈 <b>التوزيع الحالي:</b>\n";
        $text .= "• Lifetime: {$stats['lifetime_count']}\n";
        $text .= "• Timed: {$stats['timed_count']}\n\n";
        
        if ($preview['conflicts'] > 0 && !$force) {
            $text .= "⚠️ <b>تحذير:</b> يوجد {$preview['conflicts']} اشتراك lifetime لن يتم تغييره\n";
            $text .= "💡 استخدم <code>--force</code> لتغيير اشتراكات lifetime أيضاً\n\n";
        }
        
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "🔄 <b>التغيير المقترح:</b>\n";
        if ($newAccessType === 'lifetime') {
            $text .= "• النوع الجديد: ♾️ <b>Lifetime</b>\n";
        } else {
            $text .= "• النوع الجديد: ⏱️ <b>Timed ({$newAccessDays} يوم)</b>\n";
            
            // عرض تحليل الاستقطاع
            $text .= "\n📊 <b>تحليل استقطاع المدة:</b>\n";
            $text .= "• ✅ سيبقى لهم مدة: {$preview['will_have_remaining']} زبون\n";
            $text .= "• ❌ سينتهي فوراً (استهلكوا {$newAccessDays}+ يوم): {$preview['will_expire_immediately']} زبون\n";
            
            if ($preview['will_expire_immediately'] > 0) {
                $text .= "\n⚠️ <b>تحذير:</b> {$preview['will_expire_immediately']} زبون سينتهي اشتراكهم فوراً!\n";
            }
        }
        
        $text .= "\n⏱️ <b>الوقت المتوقع:</b> ~{$preview['estimated_time_seconds']} ثانية\n\n";
        
        // عرض عينة مع تفاصيل الاستقطاع
        if (!empty($samples)) {
            $text .= "━━━━━━━━━━━━━━━━━━━━\n";
            $text .= "📝 <b>عينة من السجلات (أول 5):</b>\n\n";
            
            foreach (array_slice($samples, 0, 5) as $sample) {
                $currentType = $sample['access_type'] === 'lifetime' ? '♾️' : "⏱️ {$sample['access_days']}d";
                $daysUsed = (int) ($sample['days_used'] ?? 0);
                $text .= "• TG:{$sample['telegram_id']} - {$sample['account_name']}\n";
                $text .= "  └ حالياً: {$currentType} | مستخدم: {$daysUsed} يوم\n";
                
                // حساب المتبقي بعد التغيير
                if ($newAccessType === 'timed' && $newAccessDays) {
                    $remaining = $newAccessDays - $daysUsed;
                    if ($remaining <= 0) {
                        $text .= "  └ بعد التغيير: ❌ <b>منتهي</b>\n";
                    } else {
                        $text .= "  └ بعد التغيير: ✅ {$remaining} يوم متبقي\n";
                    }
                }
            }
        }
        
        $text .= "\n━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "💡 <b>لتنفيذ التغيير فعلياً:</b>\n";
        $text .= "<code>/set_users_access " . implode(',', $accountIds) . " {$newAccessType}";
        if ($newAccessDays) $text .= " {$newAccessDays}";
        if ($force) $text .= " --force";
        $text .= "</code>";
        
        Telegram::sendLongMessage($chatId, $text);
    }
    
    /**
     * تنفيذ التغيير
     */
    private static function executeChange(int $chatId, array $accountIds, string $newAccessType, ?int $newAccessDays, bool $force, bool $notify, int $fromId): void
    {
        try {
            $result = BatchAccessManager::executeUsersAccessChange(
                $accountIds,
                $newAccessType,
                $newAccessDays,
                $force,
                0,
                $fromId,
                $notify
            );
        } catch (Throwable $e) {
            Telegram::sendMessage($chatId, "❌ خطأ في التنفيذ:\n<code>" . htmlspecialchars($e->getMessage()) . "</code>");
            return;
        }
        
        if (!$result['success']) {
            Telegram::sendMessage($chatId, "❌ فشل التنفيذ: " . $result['message']);
            return;
        }
        
        $text = "✅ <b>تم تنفيذ التغيير بنجاح!</b>\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "📊 <b>النتائج:</b>\n";
        $text .= "• السجلات المحدثة: {$result['affected_count']}\n";
        $text .= "• الأخطاء: {$result['errors_count']}\n";
        $text .= "• وقت التنفيذ: {$result['execution_time_ms']}ms\n\n";
        
        if ($newAccessType === 'lifetime') {
            $text .= "🎯 <b>التغيير:</b> تم تحويل الاشتراكات إلى ♾️ Lifetime\n";
        } else {
            $text .= "🎯 <b>التغيير:</b> تم تحويل الاشتراكات إلى ⏱️ Timed ({$newAccessDays} يوم)\n";
        }
        
        if ($notify && $result['affected_count'] > 0) {
            $text .= "\n📧 تم إرسال إشعارات للزبائن المتأثرين";
        }
        
        if (!empty($result['errors'])) {
            $text .= "\n\n⚠️ <b>الأخطاء:</b>\n";
            foreach (array_slice($result['errors'], 0, 5) as $error) {
                $text .= "• User {$error['user_id']}, Account {$error['account_id']}: {$error['error']}\n";
            }
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض مساعدة الأمر
     */
    private static function showSetUsersAccessHelp(int $chatId): void
    {
        $text = "📤 <b>تغيير صلاحيات وصول الزبائن دفعة واحدة</b>\n\n";
        $text .= "<b>الاستخدام:</b>\n";
        $text .= "<code>/set_users_access [accounts] [type] [days?] [flags?]</code>\n\n";
        
        $text .= "<b>المعاملات:</b>\n";
        $text .= "• <code>accounts</code>: معرفات الحسابات (1,2,3) أو slugs أو <code>all</code>\n";
        $text .= "• <code>type</code>: <code>lifetime</code> أو <code>timed</code>\n";
        $text .= "• <code>days</code>: عدد الأيام (مطلوب إذا كان timed)\n\n";
        
        $text .= "<b>Flags:</b>\n";
        $text .= "• <code>--preview</code>: معاينة فقط بدون تنفيذ\n";
        $text .= "• <code>--force</code>: تغيير اشتراكات lifetime أيضاً\n";
        $text .= "• <code>--notify</code>: إرسال إشعارات للزبائن\n\n";
        
        $text .= "<b>أمثلة:</b>\n";
        $text .= "<code>/set_users_access vip-acc timed 30 --preview</code>\n";
        $text .= "<code>/set_users_access 1,2,3 lifetime --force</code>\n";
        $text .= "<code>/set_users_access all timed 60 --notify</code>";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * ضبط الوصول الافتراضي للحساب
     * /set_default_account_access <account|all> <inherit_global|lifetime|timed> [days]
     */
    public static function cmdSetDefaultAccountAccess(int $chatId, string $args, int $fromId): void
    {
        $parts = preg_split('/\s+/', trim($args));
        
        if (count($parts) < 2) {
            self::showSetDefaultAccountAccessHelp($chatId);
            return;
        }
        
        $accountArg = $parts[0];
        $mode = strtolower($parts[1]);
        $days = null;
        
        if ($mode === 'timed') {
            foreach ($parts as $part) {
                if (is_numeric($part) && $part > 0) {
                    $days = (int) $part;
                    break;
                }
            }
            
            if ($days === null) {
                Telegram::sendMessage($chatId, "❌ عند اختيار <code>timed</code>، يجب تحديد عدد الأيام");
                return;
            }
        }
        
        // تحديد الحسابات
        $accountIds = [];
        if (strtolower($accountArg) === 'all') {
            $accountIds = Db::fetchColumn("SELECT id FROM accounts WHERE status = 'active'");
        } else {
            if (is_numeric($accountArg)) {
                $accountIds[] = (int) $accountArg;
            } else {
                $account = Db::fetchOne("SELECT id FROM accounts WHERE slug = ?", [$accountArg]);
                if ($account) {
                    $accountIds[] = $account['id'];
                }
            }
        }
        
        if (empty($accountIds)) {
            Telegram::sendMessage($chatId, "❌ لم يتم العثور على حسابات صالحة");
            return;
        }
        
        $success = 0;
        $failed = 0;
        
        foreach ($accountIds as $accountId) {
            $inheritGlobal = ($mode === 'inherit_global');
            $accessType = $inheritGlobal ? null : $mode;
            
            $result = BatchAccessManager::setDefaultAccountAccess(
                $accountId,
                $accessType,
                $days,
                $inheritGlobal,
                0,
                $fromId
            );
            
            if ($result) {
                $success++;
            } else {
                $failed++;
            }
        }
        
        $text = "✅ <b>تم تحديث الإعدادات الافتراضية</b>\n\n";
        $text .= "• نجح: {$success}\n";
        $text .= "• فشل: {$failed}\n\n";
        
        if ($mode === 'inherit_global') {
            $text .= "🌐 الحسابات الآن ترث الإعدادات العامة";
        } elseif ($mode === 'lifetime') {
            $text .= "♾️ الوصول الافتراضي: Lifetime";
        } else {
            $text .= "⏱️ الوصول الافتراضي: Timed ({$days} يوم)";
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض مساعدة set_default_account_access
     */
    private static function showSetDefaultAccountAccessHelp(int $chatId): void
    {
        $text = "⚙️ <b>ضبط الوصول الافتراضي للحساب</b>\n\n";
        $text .= "<b>الاستخدام:</b>\n";
        $text .= "<code>/set_default_account_access [account] [mode] [days?]</code>\n\n";
        
        $text .= "<b>المعاملات:</b>\n";
        $text .= "• <code>account</code>: ID أو slug أو <code>all</code>\n";
        $text .= "• <code>mode</code>: <code>inherit_global</code> | <code>lifetime</code> | <code>timed</code>\n";
        $text .= "• <code>days</code>: عدد الأيام (مطلوب إذا كان timed)\n\n";
        
        $text .= "<b>أمثلة:</b>\n";
        $text .= "<code>/set_default_account_access vip-acc timed 30</code>\n";
        $text .= "<code>/set_default_account_access all inherit_global</code>\n";
        $text .= "<code>/set_default_account_access 5 lifetime</code>";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض الوصول الافتراضي للحساب
     */
    public static function cmdGetDefaultAccountAccess(int $chatId, string $args): void
    {
        $accountArg = trim($args);
        
        if (empty($accountArg)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام: <code>/get_default_account_access [account|all]</code>");
            return;
        }
        
        if (strtolower($accountArg) === 'all') {
            self::showAllDefaultAccess($chatId);
        } else {
            self::showSingleDefaultAccess($chatId, $accountArg);
        }
    }
    
    /**
     * عرض الوصول الافتراضي لحساب واحد
     */
    private static function showSingleDefaultAccess(int $chatId, string $accountArg): void
    {
        $accountId = null;
        if (is_numeric($accountArg)) {
            $accountId = (int) $accountArg;
        } else {
            $account = Db::fetchOne("SELECT id FROM accounts WHERE slug = ?", [$accountArg]);
            if ($account) {
                $accountId = $account['id'];
            }
        }
        
        if (!$accountId) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود");
            return;
        }
        
        $info = BatchAccessManager::getDefaultAccountAccess($accountId);
        
        if (!$info) {
            Telegram::sendMessage($chatId, "❌ فشل جلب المعلومات");
            return;
        }
        
        $text = "⚙️ <b>الوصول الافتراضي - {$info['name']}</b>\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "📦 <b>معلومات الحساب:</b>\n";
        $text .= "• ID: {$info['id']}\n";
        $text .= "• Slug: <code>{$info['slug']}</code>\n\n";
        
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "🎯 <b>الإعدادات الحالية:</b>\n";
        
        if ($info['default_inherit_global']) {
            $text .= "• المصدر: 🌐 <b>الإعدادات العامة</b>\n";
            $text .= "• النوع الفعلي: ";
            if ($info['effective_access_type'] === 'lifetime') {
                $text .= "♾️ Lifetime\n";
            } else {
                $text .= "⏱️ Timed ({$info['effective_access_days']} يوم)\n";
            }
        } else {
            $text .= "• المصدر: 📦 <b>إعدادات الحساب</b>\n";
            $text .= "• النوع: ";
            if ($info['default_access_type'] === 'lifetime') {
                $text .= "♾️ Lifetime\n";
            } else {
                $text .= "⏱️ Timed ({$info['default_access_days']} يوم)\n";
            }
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض الوصول الافتراضي لجميع الحسابات
     */
    private static function showAllDefaultAccess(int $chatId): void
    {
        $accounts = Db::fetchAll(
            "SELECT id, slug, name, default_inherit_global, default_access_type, default_access_days
             FROM accounts
             ORDER BY name ASC
             LIMIT 50"
        );
        
        if (empty($accounts)) {
            Telegram::sendMessage($chatId, "📭 لا توجد حسابات");
            return;
        }
        
        $text = "⚙️ <b>الوصول الافتراضي - جميع الحسابات</b>\n\n";
        
        foreach ($accounts as $acc) {
            $text .= "📦 <b>{$acc['name']}</b> (<code>{$acc['slug']}</code>)\n";
            
            if ($acc['default_inherit_global']) {
                $text .= "   └ 🌐 يرث من الإعدادات العامة\n";
            } else {
                if ($acc['default_access_type'] === 'lifetime') {
                    $text .= "   └ ♾️ Lifetime\n";
                } else {
                    $text .= "   └ ⏱️ Timed ({$acc['default_access_days']} يوم)\n";
                }
            }
            $text .= "\n";
        }
        
        Telegram::sendLongMessage($chatId, $text);
    }
    
    /**
     * قائمة الحسابات التي ترث الإعدادات العامة
     */
    public static function cmdListAccountsWithDefaultInherit(int $chatId, string $args): void
    {
        $page = (int) trim($args) ?: 1;
        
        $result = BatchAccessManager::listAccountsWithDefaultInherit($page, 20);
        
        if ($result['total'] === 0) {
            Telegram::sendMessage($chatId, "📭 لا توجد حسابات ترث الإعدادات العامة");
            return;
        }
        
        $text = "🌐 <b>الحسابات التي ترث الإعدادات العامة</b>\n\n";
        $text .= "📊 الصفحة {$result['page']} من {$result['total_pages']} (إجمالي: {$result['total']})\n\n";
        
        foreach ($result['accounts'] as $acc) {
            $text .= "📦 <b>{$acc['name']}</b> (<code>{$acc['slug']}</code>)\n";
            $text .= "   └ 👥 {$acc['active_users_count']} زبون نشط\n\n";
        }
        
        if ($result['page'] < $result['total_pages']) {
            $nextPage = $result['page'] + 1;
            $text .= "📄 الصفحة التالية: <code>/list_accounts_with_default_inherit {$nextPage}</code>";
        }
        
        Telegram::sendLongMessage($chatId, $text);
    }
    
    /**
     * حالة عملية دفعية
     */
    public static function cmdBatchOperationStatus(int $chatId, string $args): void
    {
        $operationId = trim($args);
        
        if (empty($operationId)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام: <code>/batch_operation_status [operation_id]</code>");
            return;
        }
        
        $operation = Db::fetchOne(
            "SELECT * FROM batch_operations WHERE operation_id = ?",
            [$operationId]
        );
        
        if (!$operation) {
            Telegram::sendMessage($chatId, "❌ العملية غير موجودة");
            return;
        }
        
        $statusIcons = [
            'pending' => '⏳',
            'running' => '▶️',
            'completed' => '✅',
            'failed' => '❌',
            'cancelled' => '🚫',
        ];
        
        $icon = $statusIcons[$operation['status']] ?? '❓';
        
        $text = "{$icon} <b>حالة العملية الدفعية</b>\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "🆔 <b>المعرف:</b> <code>{$operation['operation_id']}</code>\n";
        $text .= "📋 <b>النوع:</b> {$operation['operation_type']}\n";
        $text .= "📊 <b>الحالة:</b> {$icon} {$operation['status']}\n\n";
        
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "📈 <b>التقدم:</b>\n";
        $text .= "• الإجمالي: {$operation['total_records']}\n";
        $text .= "• المعالج: {$operation['processed_records']}\n";
        $text .= "• النجاح: {$operation['success_count']}\n";
        $text .= "• الفشل: {$operation['failed_count']}\n";
        $text .= "• النسبة: {$operation['progress_percent']}%\n\n";
        
        if ($operation['started_at']) {
            $text .= "⏰ <b>بدأت:</b> {$operation['started_at']}\n";
        }
        
        if ($operation['completed_at']) {
            $text .= "✅ <b>انتهت:</b> {$operation['completed_at']}\n";
        }
        
        if ($operation['error_log']) {
            $text .= "\n⚠️ <b>الأخطاء:</b>\n{$operation['error_log']}";
        }
        
        Telegram::sendMessage($chatId, $text);
    }
}
