<?php
/**
 * نظام الإعدادات المخصصة (Settings System)
 * 
 * يدعم الإعدادات على مستويات متعددة:
 * - global: إعدادات عامة للنظام
 * - account: إعدادات خاصة بحساب
 * - superdist: إعدادات خاصة بموزع
 * - user: إعدادات خاصة بمستخدم
 * - user_account: إعدادات خاصة بمستخدم+حساب
 * 
 * الأولوية: user_account > user > superdist > account > global
 */

class Settings
{
    // المستويات المدعومة
    const LEVEL_GLOBAL = 'global';
    const LEVEL_ACCOUNT = 'account';
    const LEVEL_SUPERDIST = 'superdist';
    const LEVEL_USER = 'user';
    const LEVEL_USER_ACCOUNT = 'user_account';
    
    // الإعدادات المعروفة مع قيمها الافتراضية
    private static array $defaults = [
        'activation_default_type' => 'lifetime',
        'activation_default_days' => 30,
        'voucher_redeem_allowed' => true,
        'broadcast_allowed_types' => ['text', 'stickers', 'photos', 'videos', 'documents', 'audio'],
        'max_codes_per_generation' => 100,
        'code_expiry_days' => 365,
        'daily_activation_limit' => null,
        'weekly_activation_limit' => null,
        'monthly_activation_limit' => null,
    ];
    
    /**
     * جلب إعداد مع تطبيق الأولوية
     * 
     * @param string $key مفتاح الإعداد
     * @param int|null $userId معرف المستخدم (اختياري)
     * @param int|null $accountId معرف الحساب (اختياري)
     * @param int|null $superdistId معرف الموزع (اختياري)
     * @return mixed قيمة الإعداد
     */
    public static function get(string $key, ?int $userId = null, ?int $accountId = null, ?int $superdistId = null)
    {
        // ترتيب الأولوية من الأعلى للأدنى
        $levelsToCheck = [];
        
        // user_account (الأعلى)
        if ($userId && $accountId) {
            $levelsToCheck[] = ['level' => self::LEVEL_USER_ACCOUNT, 'id' => $userId, 'id2' => $accountId];
        }
        
        // user
        if ($userId) {
            $levelsToCheck[] = ['level' => self::LEVEL_USER, 'id' => $userId, 'id2' => null];
        }
        
        // superdist
        if ($superdistId) {
            $levelsToCheck[] = ['level' => self::LEVEL_SUPERDIST, 'id' => $superdistId, 'id2' => null];
        }
        
        // account
        if ($accountId) {
            $levelsToCheck[] = ['level' => self::LEVEL_ACCOUNT, 'id' => $accountId, 'id2' => null];
        }
        
        // global (الأدنى)
        $levelsToCheck[] = ['level' => self::LEVEL_GLOBAL, 'id' => null, 'id2' => null];
        
        // البحث حسب الأولوية
        foreach ($levelsToCheck as $check) {
            $value = self::getFromLevel($key, $check['level'], $check['id'], $check['id2']);
            if ($value !== null) {
                return $value;
            }
        }
        
        // القيمة الافتراضية
        return self::$defaults[$key] ?? null;
    }
    
    /**
     * جلب إعداد من مستوى محدد
     */
    public static function getFromLevel(string $key, string $level, $levelId = null, $levelId2 = null)
    {
        $sql = "SELECT setting_value, value_type FROM system_settings 
                WHERE setting_key = ? AND level = ?";
        $params = [$key, $level];
        
        if ($levelId !== null) {
            $sql .= " AND level_id = ?";
            $params[] = (string)$levelId;
        } else {
            $sql .= " AND level_id IS NULL";
        }
        
        if ($levelId2 !== null) {
            $sql .= " AND level_id_secondary = ?";
            $params[] = (string)$levelId2;
        } else {
            $sql .= " AND level_id_secondary IS NULL";
        }
        
        $row = Db::fetchOne($sql, $params);
        
        if (!$row) {
            return null;
        }
        
        return self::castValue($row['setting_value'], $row['value_type']);
    }
    
    /**
     * تعيين إعداد
     */
    public static function set(string $key, $value, string $level, $levelId = null, $levelId2 = null, ?int $updatedBy = null): bool
    {
        // تحديد نوع القيمة
        $valueType = self::detectValueType($value);
        $stringValue = self::valueToString($value, $valueType);
        
        // التحقق من وجود الإعداد
        $existing = self::getFromLevel($key, $level, $levelId, $levelId2);
        
        if ($existing !== null) {
            // تحديث
            $sql = "UPDATE system_settings SET setting_value = ?, value_type = ?, updated_by = ?, updated_at = NOW()
                    WHERE setting_key = ? AND level = ?";
            $params = [$stringValue, $valueType, $updatedBy, $key, $level];
            
            if ($levelId !== null) {
                $sql .= " AND level_id = ?";
                $params[] = (string)$levelId;
            } else {
                $sql .= " AND level_id IS NULL";
            }
            
            if ($levelId2 !== null) {
                $sql .= " AND level_id_secondary = ?";
                $params[] = (string)$levelId2;
            } else {
                $sql .= " AND level_id_secondary IS NULL";
            }
            
            Db::execute($sql, $params);
        } else {
            // إدراج جديد
            Db::insert('system_settings', [
                'level' => $level,
                'level_id' => $levelId !== null ? (string)$levelId : null,
                'level_id_secondary' => $levelId2 !== null ? (string)$levelId2 : null,
                'setting_key' => $key,
                'setting_value' => $stringValue,
                'value_type' => $valueType,
                'updated_by' => $updatedBy,
            ]);
        }
        
        Logger::event('setting_changed', 'admin', $updatedBy, 'setting', null, [
            'key' => $key,
            'level' => $level,
            'level_id' => $levelId,
            'level_id_secondary' => $levelId2,
            'new_value' => $stringValue,
        ]);
        
        return true;
    }
    
    /**
     * حذف إعداد
     */
    public static function delete(string $key, string $level, $levelId = null, $levelId2 = null): bool
    {
        $sql = "DELETE FROM system_settings WHERE setting_key = ? AND level = ?";
        $params = [$key, $level];
        
        if ($levelId !== null) {
            $sql .= " AND level_id = ?";
            $params[] = (string)$levelId;
        } else {
            $sql .= " AND level_id IS NULL";
        }
        
        if ($levelId2 !== null) {
            $sql .= " AND level_id_secondary = ?";
            $params[] = (string)$levelId2;
        } else {
            $sql .= " AND level_id_secondary IS NULL";
        }
        
        return Db::execute($sql, $params) > 0;
    }
    
    /**
     * جلب جميع الإعدادات لمستوى معين
     */
    public static function getAllForLevel(string $level, $levelId = null, $levelId2 = null): array
    {
        $sql = "SELECT setting_key, setting_value, value_type FROM system_settings WHERE level = ?";
        $params = [$level];
        
        if ($levelId !== null) {
            $sql .= " AND level_id = ?";
            $params[] = (string)$levelId;
        } else {
            $sql .= " AND level_id IS NULL";
        }
        
        if ($levelId2 !== null) {
            $sql .= " AND level_id_secondary = ?";
            $params[] = (string)$levelId2;
        } else {
            $sql .= " AND level_id_secondary IS NULL";
        }
        
        $rows = Db::fetchAll($sql, $params);
        
        $settings = [];
        foreach ($rows as $row) {
            $settings[$row['setting_key']] = self::castValue($row['setting_value'], $row['value_type']);
        }
        
        return $settings;
    }
    
    /**
     * جلب جميع الإعدادات (للعرض)
     */
    public static function getAll(): array
    {
        $rows = Db::fetchAll("SELECT * FROM system_settings ORDER BY level, level_id, setting_key");
        
        $result = [];
        foreach ($rows as $row) {
            $row['parsed_value'] = self::castValue($row['setting_value'], $row['value_type']);
            $result[] = $row;
        }
        
        return $result;
    }
    
    /**
     * تحويل القيمة من string للنوع المناسب
     */
    private static function castValue(string $value, string $type)
    {
        switch ($type) {
            case 'int':
                return (int)$value;
            case 'float':
                return (float)$value;
            case 'bool':
                return in_array(strtolower($value), ['true', '1', 'yes']);
            case 'json':
            case 'array':
                return json_decode($value, true) ?? [];
            default:
                return $value;
        }
    }
    
    /**
     * تحويل القيمة إلى string
     */
    private static function valueToString($value, string $type): string
    {
        switch ($type) {
            case 'bool':
                return $value ? 'true' : 'false';
            case 'json':
            case 'array':
                return json_encode($value, JSON_UNESCAPED_UNICODE);
            default:
                return (string)$value;
        }
    }
    
    /**
     * تحديد نوع القيمة
     */
    private static function detectValueType($value): string
    {
        if (is_bool($value)) {
            return 'bool';
        }
        if (is_int($value)) {
            return 'int';
        }
        if (is_float($value)) {
            return 'float';
        }
        if (is_array($value)) {
            return 'json';
        }
        return 'string';
    }
    
    /**
     * الإعدادات الافتراضية
     */
    public static function getDefaults(): array
    {
        return self::$defaults;
    }
    
    /**
     * التحقق من صحة مفتاح الإعداد
     */
    public static function isValidKey(string $key): bool
    {
        return array_key_exists($key, self::$defaults) || preg_match('/^[a-z][a-z0-9_]*$/', $key);
    }
    
    /**
     * التحقق من صحة المستوى
     */
    public static function isValidLevel(string $level): bool
    {
        return in_array($level, [
            self::LEVEL_GLOBAL,
            self::LEVEL_ACCOUNT,
            self::LEVEL_SUPERDIST,
            self::LEVEL_USER,
            self::LEVEL_USER_ACCOUNT,
        ]);
    }
}
