<?php
/**
 * معالجة الأكواد ضمن نافذة زمنية (Process Codes Window)
 * 
 * يوفر آلية لمعالجة أكواد التفعيل بناءً على:
 * - نافذة زمنية (since: منذ تاريخ معين)
 * - عدد محدد (last_n: آخر N كود)
 * 
 * الإجراءات المتاحة:
 * - expire_unused: تغيير الحالة إلى expired
 * - disable_unused: تغيير الحالة إلى disabled
 * - report_only: تقرير فقط بدون تغيير
 */

class ProcessCodesWindow
{
    const MODE_SINCE = 'since';
    const MODE_LAST_N = 'last_n';
    
    const ACTION_EXPIRE = 'expire_unused';
    const ACTION_DISABLE = 'disable_unused';
    const ACTION_REPORT = 'report_only';
    
    const REPORTS_DIR = __DIR__ . '/../reports/process_jobs';
    
    private int $jobId;
    private string $mode;
    private string $modeValue;
    private string $action;
    private ?int $scopeAccountId;
    private ?int $scopeSuperdistId;
    private int $initiatedBy;
    
    private int $totalCodes = 0;
    private int $processedCodes = 0;
    private int $affectedCodes = 0;
    private int $skippedCodes = 0;
    private array $details = [];
    
    /**
     * إنشاء وتنفيذ مهمة جديدة
     */
    public static function run(
        string $mode,
        string $modeValue,
        string $action,
        ?int $accountId,
        ?int $superdistId,
        int $initiatedBy
    ): array {
        $processor = new self($mode, $modeValue, $action, $accountId, $superdistId, $initiatedBy);
        return $processor->execute();
    }
    
    private function __construct(
        string $mode,
        string $modeValue,
        string $action,
        ?int $accountId,
        ?int $superdistId,
        int $initiatedBy
    ) {
        $this->mode = $mode;
        $this->modeValue = $modeValue;
        $this->action = $action;
        $this->scopeAccountId = $accountId;
        $this->scopeSuperdistId = $superdistId;
        $this->initiatedBy = $initiatedBy;
        
        // إنشاء مجلد التقارير
        if (!is_dir(self::REPORTS_DIR)) {
            mkdir(self::REPORTS_DIR, 0755, true);
        }
        
        // إنشاء سجل المهمة
        $this->createJob();
    }
    
    /**
     * إنشاء سجل المهمة
     */
    private function createJob(): void
    {
        $this->jobId = Db::insert('process_jobs', [
            'job_type' => 'codes_window',
            'mode' => $this->mode,
            'mode_value' => $this->modeValue,
            'action' => $this->action,
            'scope_account_id' => $this->scopeAccountId,
            'scope_superdist_id' => $this->scopeSuperdistId,
            'scope_json' => json_encode([
                'account_id' => $this->scopeAccountId,
                'superdist_id' => $this->scopeSuperdistId,
            ]),
            'initiated_by' => $this->initiatedBy,
            'status' => 'pending',
        ]);
    }
    
    /**
     * تنفيذ المهمة
     */
    private function execute(): array
    {
        // تحديث الحالة إلى running
        $this->updateJobStatus('running');
        
        try {
            // بناء الاستعلام
            $codes = $this->fetchCodes();
            $this->totalCodes = count($codes);
            
            // تحديث العدد الإجمالي
            Db::update('process_jobs', ['total_codes' => $this->totalCodes], 'id = ?', [$this->jobId]);
            
            // معالجة الأكواد
            foreach ($codes as $code) {
                $this->processCode($code);
                $this->processedCodes++;
                
                // تحديث التقدم كل 100 كود
                if ($this->processedCodes % 100 === 0) {
                    $this->updateProgress();
                }
            }
            
            // إنشاء التقارير
            $reportPath = $this->generateReport();
            $csvPath = $this->generateCsv();
            
            // تحديث المهمة
            $this->completeJob($reportPath, $csvPath);
            
            return [
                'success' => true,
                'job_id' => $this->jobId,
                'total' => $this->totalCodes,
                'processed' => $this->processedCodes,
                'affected' => $this->affectedCodes,
                'skipped' => $this->skippedCodes,
                'report_path' => $reportPath,
                'csv_path' => $csvPath,
            ];
            
        } catch (\Exception $e) {
            $this->failJob($e->getMessage());
            return [
                'success' => false,
                'job_id' => $this->jobId,
                'error' => $e->getMessage(),
            ];
        }
    }
    
    /**
     * جلب الأكواد حسب المعايير
     */
    private function fetchCodes(): array
    {
        $sql = "SELECT ac.*, a.slug as account_slug 
                FROM activation_codes ac
                LEFT JOIN accounts a ON a.id = ac.account_id
                WHERE ac.status IN ('unused', 'available')";
        $params = [];
        
        // فلتر الحساب
        if ($this->scopeAccountId) {
            $sql .= " AND ac.account_id = ?";
            $params[] = $this->scopeAccountId;
        }
        
        // فلتر الموزع
        if ($this->scopeSuperdistId) {
            $sql .= " AND ac.superdist_id = ?";
            $params[] = $this->scopeSuperdistId;
        }
        
        // فلتر النافذة الزمنية
        if ($this->mode === self::MODE_SINCE) {
            $date = $this->parseDateValue($this->modeValue);
            $sql .= " AND ac.created_at >= ?";
            $params[] = $date;
        }
        
        $sql .= " ORDER BY ac.created_at DESC";
        
        // فلتر العدد
        if ($this->mode === self::MODE_LAST_N) {
            $sql .= " LIMIT " . (int)$this->modeValue;
        }
        
        return Db::fetchAll($sql, $params);
    }
    
    /**
     * تحويل قيمة التاريخ
     */
    private function parseDateValue(string $value): string
    {
        // صيغة "30d" للأيام الماضية
        if (preg_match('/^(\d+)d$/', $value, $matches)) {
            $days = (int)$matches[1];
            return date('Y-m-d H:i:s', strtotime("-{$days} days"));
        }
        
        // صيغة تاريخ YYYY-MM-DD
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value . ' 00:00:00';
        }
        
        // محاولة تحويل أي صيغة
        $timestamp = strtotime($value);
        if ($timestamp !== false) {
            return date('Y-m-d H:i:s', $timestamp);
        }
        
        throw new \InvalidArgumentException("صيغة التاريخ غير صالحة: {$value}");
    }
    
    /**
     * معالجة كود واحد
     */
    private function processCode(array $code): void
    {
        $oldStatus = $code['status'];
        $newStatus = $oldStatus;
        $actionTaken = 'none';
        $reason = '';
        
        // تحديد الإجراء
        if ($this->action === self::ACTION_EXPIRE) {
            $newStatus = 'expired';
            $actionTaken = 'expired';
            $reason = 'معالجة دفعية - انتهاء الصلاحية';
        } elseif ($this->action === self::ACTION_DISABLE) {
            $newStatus = 'disabled';
            $actionTaken = 'disabled';
            $reason = 'معالجة دفعية - تعطيل';
        } elseif ($this->action === self::ACTION_REPORT) {
            $actionTaken = 'reported';
            $reason = 'تقرير فقط';
        }
        
        // تنفيذ التغيير (إلا إذا كان تقرير فقط)
        if ($this->action !== self::ACTION_REPORT && $newStatus !== $oldStatus) {
            Db::update('activation_codes', ['status' => $newStatus], 'id = ?', [$code['id']]);
            $this->affectedCodes++;
        } else {
            $this->skippedCodes++;
        }
        
        // حفظ التفاصيل
        $detail = [
            'job_id' => $this->jobId,
            'activation_code_id' => $code['id'],
            'code' => $code['code'],
            'account_slug' => $code['account_slug'] ?? null,
            'superdist_id' => $code['superdist_id'] ?? null,
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'action_taken' => $actionTaken,
            'reason' => $reason,
        ];
        
        Db::insert('process_job_details', $detail);
        $this->details[] = $detail;
    }
    
    /**
     * تحديث التقدم
     */
    private function updateProgress(): void
    {
        $percent = $this->totalCodes > 0 
            ? (int)(($this->processedCodes / $this->totalCodes) * 100)
            : 0;
        
        Db::update('process_jobs', [
            'processed_codes' => $this->processedCodes,
            'progress_percent' => $percent,
        ], 'id = ?', [$this->jobId]);
    }
    
    /**
     * تحديث حالة المهمة
     */
    private function updateJobStatus(string $status): void
    {
        $data = ['status' => $status];
        
        if ($status === 'running') {
            $data['started_at'] = date('Y-m-d H:i:s');
        }
        
        Db::update('process_jobs', $data, 'id = ?', [$this->jobId]);
    }
    
    /**
     * إكمال المهمة
     */
    private function completeJob(string $reportPath, string $csvPath): void
    {
        Db::update('process_jobs', [
            'status' => 'completed',
            'processed_codes' => $this->processedCodes,
            'affected_codes' => $this->affectedCodes,
            'skipped_codes' => $this->skippedCodes,
            'progress_percent' => 100,
            'result_summary_json' => json_encode([
                'total' => $this->totalCodes,
                'processed' => $this->processedCodes,
                'affected' => $this->affectedCodes,
                'skipped' => $this->skippedCodes,
            ]),
            'report_path' => $reportPath,
            'csv_path' => $csvPath,
            'finished_at' => date('Y-m-d H:i:s'),
        ], 'id = ?', [$this->jobId]);
        
        // تسجيل الحدث
        Logger::event('process_codes_completed', 'admin', $this->initiatedBy, 'process_job', $this->jobId, [
            'mode' => $this->mode,
            'action' => $this->action,
            'affected' => $this->affectedCodes,
        ]);
    }
    
    /**
     * فشل المهمة
     */
    private function failJob(string $error): void
    {
        Db::update('process_jobs', [
            'status' => 'failed',
            'result_summary_json' => json_encode(['error' => $error]),
            'finished_at' => date('Y-m-d H:i:s'),
        ], 'id = ?', [$this->jobId]);
        
        Logger::error('Process codes job failed', [
            'job_id' => $this->jobId,
            'error' => $error,
        ]);
    }
    
    /**
     * إنشاء تقرير JSON
     */
    private function generateReport(): string
    {
        $filename = date('Y-m-d_H-i-s') . "_job_{$this->jobId}.json";
        $path = self::REPORTS_DIR . '/' . $filename;
        
        $report = [
            'job_id' => $this->jobId,
            'generated_at' => date('Y-m-d H:i:s'),
            'mode' => $this->mode,
            'mode_value' => $this->modeValue,
            'action' => $this->action,
            'scope' => [
                'account_id' => $this->scopeAccountId,
                'superdist_id' => $this->scopeSuperdistId,
            ],
            'summary' => [
                'total_codes' => $this->totalCodes,
                'processed_codes' => $this->processedCodes,
                'affected_codes' => $this->affectedCodes,
                'skipped_codes' => $this->skippedCodes,
            ],
            'details' => $this->details,
        ];
        
        file_put_contents($path, json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        return $path;
    }
    
    /**
     * إنشاء تقرير CSV
     */
    private function generateCsv(): string
    {
        $filename = date('Y-m-d_H-i-s') . "_job_{$this->jobId}.csv";
        $path = self::REPORTS_DIR . '/' . $filename;
        
        $fp = fopen($path, 'w');
        
        // BOM for UTF-8
        fwrite($fp, "\xEF\xBB\xBF");
        
        // Header
        fputcsv($fp, ['code', 'account_slug', 'superdist_id', 'old_status', 'new_status', 'action_taken', 'reason']);
        
        // Data
        foreach ($this->details as $detail) {
            fputcsv($fp, [
                $detail['code'],
                $detail['account_slug'] ?? '',
                $detail['superdist_id'] ?? '',
                $detail['old_status'],
                $detail['new_status'],
                $detail['action_taken'],
                $detail['reason'],
            ]);
        }
        
        fclose($fp);
        
        return $path;
    }
    
    /**
     * جلب معلومات مهمة
     */
    public static function getJob(int $jobId): ?array
    {
        return Db::fetchOne("SELECT * FROM process_jobs WHERE id = ?", [$jobId]);
    }
    
    /**
     * جلب قائمة المهام
     */
    public static function getJobs(int $limit = 20): array
    {
        return Db::fetchAll(
            "SELECT * FROM process_jobs ORDER BY created_at DESC LIMIT ?",
            [$limit]
        );
    }
    
    /**
     * جلب تفاصيل مهمة
     */
    public static function getJobDetails(int $jobId, int $limit = 100): array
    {
        return Db::fetchAll(
            "SELECT * FROM process_job_details WHERE job_id = ? ORDER BY id LIMIT ?",
            [$jobId, $limit]
        );
    }
}
