<?php
/**
 * ============================================================
 * مكتبة الحدود (Limits System)
 * ============================================================
 * 
 * تدير نظام الحدود بأربع طبقات:
 * 1. limits_user_account (أعلى أولوية)
 * 2. limits_user
 * 3. limits_superdist
 * 4. limits_global (أدنى أولوية)
 * 
 * الاستخدام:
 *   $result = Limits::checkVerificationLimit($userId, $accountId, $superdistId);
 *   if (!$result['allowed']) { echo $result['reason']; }
 */

class Limits
{
    /** @var array الحدود الافتراضية من الملف */
    private static array $defaultLimits = [];
    
    /**
     * تهيئة المكتبة
     * 
     * @param array $limitsConfig إعدادات الحدود من limits.php
     */
    public static function init(array $limitsConfig): void
    {
        self::$defaultLimits = $limitsConfig;
    }
    
    /**
     * التحقق من حدود طلب أكواد التحقق
     * 
     * @param int $userId معرف المستخدم
     * @param int $accountId معرف الحساب
     * @param int|null $superdistId معرف السوبر موزع المصدر
     * @return array نتيجة التحقق بالشكل التالي:
     * 
     * ============================================================
     * هيكل النتيجة (Return Structure):
     * ============================================================
     * [
     *   'allowed'    => bool,      // هل مسموح بالطلب؟
     *   'reason'     => string,    // سبب الرفض أو 'OK' للسماح
     *   'layer'      => string,    // الطبقة المسؤولة عن القرار
     *   'limit_type' => string,    // نوع الحد (daily/weekly/monthly)
     *   'limit'      => int,       // قيمة الحد
     *   'used'       => int,       // الاستخدام الحالي
     * ]
     * 
     * ============================================================
     * قيم reason المحتملة:
     * ============================================================
     * - 'OK'                    : مسموح
     * - 'DAILY_LIMIT_EXCEEDED'  : تجاوز الحد اليومي
     * - 'WEEKLY_LIMIT_EXCEEDED' : تجاوز الحد الأسبوعي
     * - 'MONTHLY_LIMIT_EXCEEDED': تجاوز الحد الشهري
     * - 'BLOCKED'               : محظور (حد = 0)
     * 
     * ============================================================
     * قيم layer المحتملة:
     * ============================================================
     * - 'user_account' : حد خاص بالمستخدم + الحساب
     * - 'user'         : حد خاص بالمستخدم
     * - 'superdist'    : حد خاص بالسوبر موزع
     * - 'global'       : الحد العام
     */
    public static function checkVerificationLimit(int $userId, int $accountId, ?int $superdistId = null): array
    {
        // جلب الحدود الفعّالة مع معلومات الطبقة
        $limitsInfo = self::getEffectiveVerificationLimitsWithLayer($userId, $accountId, $superdistId);
        $limits = $limitsInfo['limits'];
        $layers = $limitsInfo['layers']; // أي طبقة حددت كل حد
        
        // جلب الاستخدام الحالي
        $usage = self::getVerificationUsage($userId, $accountId);
        
        // التحقق من الحد اليومي
        if ($limits['daily'] !== null && $limits['daily'] > 0) {
            if ($usage['daily'] >= $limits['daily']) {
                return [
                    'allowed'    => false,
                    'reason'     => 'DAILY_LIMIT_EXCEEDED',
                    'layer'      => $layers['daily'],
                    'limit_type' => 'daily',
                    'limit'      => $limits['daily'],
                    'used'       => $usage['daily'],
                ];
            }
        } elseif ($limits['daily'] === 0) {
            return [
                'allowed'    => false,
                'reason'     => 'BLOCKED',
                'layer'      => $layers['daily'],
                'limit_type' => 'daily',
                'limit'      => 0,
                'used'       => $usage['daily'],
            ];
        }
        
        // التحقق من الحد الأسبوعي
        if ($limits['weekly'] !== null && $limits['weekly'] > 0) {
            if ($usage['weekly'] >= $limits['weekly']) {
                return [
                    'allowed'    => false,
                    'reason'     => 'WEEKLY_LIMIT_EXCEEDED',
                    'layer'      => $layers['weekly'],
                    'limit_type' => 'weekly',
                    'limit'      => $limits['weekly'],
                    'used'       => $usage['weekly'],
                ];
            }
        } elseif ($limits['weekly'] === 0) {
            return [
                'allowed'    => false,
                'reason'     => 'BLOCKED',
                'layer'      => $layers['weekly'],
                'limit_type' => 'weekly',
                'limit'      => 0,
                'used'       => $usage['weekly'],
            ];
        }
        
        // التحقق من الحد الشهري
        if ($limits['monthly'] !== null && $limits['monthly'] > 0) {
            if ($usage['monthly'] >= $limits['monthly']) {
                return [
                    'allowed'    => false,
                    'reason'     => 'MONTHLY_LIMIT_EXCEEDED',
                    'layer'      => $layers['monthly'],
                    'limit_type' => 'monthly',
                    'limit'      => $limits['monthly'],
                    'used'       => $usage['monthly'],
                ];
            }
        } elseif ($limits['monthly'] === 0) {
            return [
                'allowed'    => false,
                'reason'     => 'BLOCKED',
                'layer'      => $layers['monthly'],
                'limit_type' => 'monthly',
                'limit'      => 0,
                'used'       => $usage['monthly'],
            ];
        }
        
        return [
            'allowed'    => true,
            'reason'     => 'OK',
            'layer'      => 'global',
            'limit_type' => null,
            'limit'      => null,
            'used'       => null,
            'limits'     => $limits,
            'usage'      => $usage,
        ];
    }
    
    /**
     * جلب الحدود الفعّالة لطلب أكواد التحقق
     * 
     * الأولوية: user_account > user > superdist > global
     * 
     * @param int $userId
     * @param int $accountId
     * @param int|null $superdistId
     * @return array ['daily' => int|null, 'weekly' => int|null, 'monthly' => int|null]
     */
    public static function getEffectiveVerificationLimits(int $userId, int $accountId, ?int $superdistId = null): array
    {
        $result = self::getEffectiveVerificationLimitsWithLayer($userId, $accountId, $superdistId);
        return $result['limits'];
    }
    
    /**
     * جلب الحدود الفعّالة مع معلومات الطبقة المسؤولة عن كل حد
     * 
     * @param int $userId
     * @param int $accountId
     * @param int|null $superdistId
     * @return array ['limits' => [...], 'layers' => ['daily' => 'layer', ...]]
     */
    public static function getEffectiveVerificationLimitsWithLayer(int $userId, int $accountId, ?int $superdistId = null): array
    {
        // تهيئة الحدود والطبقات
        $limits = ['daily' => null, 'weekly' => null, 'monthly' => null];
        $layers = ['daily' => 'global', 'weekly' => 'global', 'monthly' => 'global'];
        
        // الطبقة 4: الحدود العامة (الأدنى أولوية)
        $globalLimits = self::getGlobalLimits();
        foreach (['daily', 'weekly', 'monthly'] as $type) {
            if ($globalLimits[$type] !== null) {
                $limits[$type] = $globalLimits[$type];
                $layers[$type] = 'global';
            }
        }
        
        // الطبقة 3.5: حدود الحساب (بين global و superdist)
        $accountLimits = self::getAccountLimits($accountId);
        foreach (['daily', 'weekly', 'monthly'] as $type) {
            if ($accountLimits[$type] !== null) {
                $limits[$type] = $accountLimits[$type];
                $layers[$type] = 'account';
            }
        }
        
        // الطبقة 3: حدود السوبر موزع
        if ($superdistId !== null) {
            $sdLimits = self::getSuperDistLimits($superdistId);
            foreach (['daily', 'weekly', 'monthly'] as $type) {
                if ($sdLimits[$type] !== null) {
                    $limits[$type] = $sdLimits[$type];
                    $layers[$type] = 'superdist';
                }
            }
        }
        
        // الطبقة 2: حدود المستخدم
        $userLimits = self::getUserLimits($userId);
        foreach (['daily', 'weekly', 'monthly'] as $type) {
            if ($userLimits[$type] !== null) {
                $limits[$type] = $userLimits[$type];
                $layers[$type] = 'user';
            }
        }
        
        // الطبقة 1: حدود المستخدم + الحساب (أعلى أولوية)
        $userAccountLimits = self::getUserAccountLimits($userId, $accountId);
        foreach (['daily', 'weekly', 'monthly'] as $type) {
            if ($userAccountLimits[$type] !== null) {
                $limits[$type] = $userAccountLimits[$type];
                $layers[$type] = 'user_account';
            }
        }
        
        return [
            'limits' => $limits,
            'layers' => $layers,
        ];
    }
    
    /**
     * دمج طبقتين من الحدود (الثانية تُتجاوز الأولى إذا ليست NULL)
     * 
     * @param array $base الحدود الأساسية
     * @param array $override الحدود الجديدة
     * @return array
     */
    private static function mergeLimits(array $base, array $override): array
    {
        return [
            'daily'   => $override['daily'] ?? $base['daily'] ?? null,
            'weekly'  => $override['weekly'] ?? $base['weekly'] ?? null,
            'monthly' => $override['monthly'] ?? $base['monthly'] ?? null,
        ];
    }
    
    /**
     * جلب الحدود العامة
     * 
     * @return array
     */
    public static function getGlobalLimits(): array
    {
        $row = Db::fetchOne("SELECT daily_limit, weekly_limit, monthly_limit FROM limits_global WHERE limit_type = 'verification' LIMIT 1");
        
        if ($row) {
            return [
                'daily'   => (int) $row['daily_limit'],
                'weekly'  => (int) $row['weekly_limit'],
                'monthly' => (int) $row['monthly_limit'],
            ];
        }
        
        // fallback إلى الملف
        return [
            'daily'   => self::$defaultLimits['verification']['daily'] ?? 2,
            'weekly'  => self::$defaultLimits['verification']['weekly'] ?? 10,
            'monthly' => self::$defaultLimits['verification']['monthly'] ?? 30,
        ];
    }
    
    /**
     * جلب حدود الحساب
     * 
     * @param int $accountId
     * @return array
     */
    public static function getAccountLimits(int $accountId): array
    {
        $row = Db::fetchOne(
            "SELECT daily_limit, weekly_limit, monthly_limit FROM limits_account WHERE account_id = ?",
            [$accountId]
        );
        
        return [
            'daily'   => $row['daily_limit'] ?? null,
            'weekly'  => $row['weekly_limit'] ?? null,
            'monthly' => $row['monthly_limit'] ?? null,
        ];
    }
    
    /**
     * جلب حدود السوبر موزع
     * 
     * @param int $superdistId
     * @return array
     */
    public static function getSuperDistLimits(int $superdistId): array
    {
        $row = Db::fetchOne(
            "SELECT daily_limit, weekly_limit, monthly_limit FROM limits_superdist WHERE superdist_id = ?",
            [$superdistId]
        );
        
        return [
            'daily'   => $row['daily_limit'] ?? null,
            'weekly'  => $row['weekly_limit'] ?? null,
            'monthly' => $row['monthly_limit'] ?? null,
        ];
    }
    
    /**
     * جلب حدود المستخدم
     * 
     * @param int $userId
     * @return array
     */
    public static function getUserLimits(int $userId): array
    {
        $row = Db::fetchOne(
            "SELECT daily_limit, weekly_limit, monthly_limit FROM limits_user WHERE user_id = ?",
            [$userId]
        );
        
        return [
            'daily'   => $row['daily_limit'] ?? null,
            'weekly'  => $row['weekly_limit'] ?? null,
            'monthly' => $row['monthly_limit'] ?? null,
        ];
    }
    
    /**
     * جلب حدود المستخدم + الحساب
     * 
     * @param int $userId
     * @param int $accountId
     * @return array
     */
    public static function getUserAccountLimits(int $userId, int $accountId): array
    {
        $row = Db::fetchOne(
            "SELECT daily_limit, weekly_limit, monthly_limit FROM limits_user_account WHERE user_id = ? AND account_id = ?",
            [$userId, $accountId]
        );
        
        return [
            'daily'   => $row['daily_limit'] ?? null,
            'weekly'  => $row['weekly_limit'] ?? null,
            'monthly' => $row['monthly_limit'] ?? null,
        ];
    }
    
    /**
     * جلب استخدام المستخدم لأكواد التحقق
     * 
     * @param int $userId
     * @param int $accountId
     * @return array ['daily' => int, 'weekly' => int, 'monthly' => int]
     */
    public static function getVerificationUsage(int $userId, int $accountId): array
    {
        // اليوم
        $daily = (int) Db::fetchValue(
            "SELECT COUNT(*) FROM verification_logs 
             WHERE user_id = ? AND account_id = ? AND DATE(requested_at) = CURDATE()",
            [$userId, $accountId]
        );
        
        // الأسبوع (من الأحد)
        $weekly = (int) Db::fetchValue(
            "SELECT COUNT(*) FROM verification_logs 
             WHERE user_id = ? AND account_id = ? 
             AND requested_at >= DATE_SUB(CURDATE(), INTERVAL WEEKDAY(CURDATE()) DAY)",
            [$userId, $accountId]
        );
        
        // الشهر
        $monthly = (int) Db::fetchValue(
            "SELECT COUNT(*) FROM verification_logs 
             WHERE user_id = ? AND account_id = ? 
             AND YEAR(requested_at) = YEAR(CURDATE()) AND MONTH(requested_at) = MONTH(CURDATE())",
            [$userId, $accountId]
        );
        
        return [
            'daily'   => $daily,
            'weekly'  => $weekly,
            'monthly' => $monthly,
        ];
    }
    
    /**
     * تسجيل طلب كود تحقق
     * 
     * @param int $userId
     * @param int $accountId
     * @param int|null $superdistId
     * @param string|null $code
     */
    public static function logVerificationRequest(int $userId, int $accountId, ?int $superdistId = null, ?string $code = null): void
    {
        Db::insert('verification_logs', [
            'user_id'           => $userId,
            'account_id'        => $accountId,
            'superdist_id'      => $superdistId,
            'verification_code' => $code,
            'requested_at'      => date('Y-m-d H:i:s'),
        ]);
    }
    
    // ================================================================
    // حدود توليد أكواد التفعيل (للسوبر موزع)
    // ================================================================
    
    /**
     * التحقق من حدود توليد أكواد التفعيل للسوبر موزع
     * 
     * @param int $superdistId
     * @param int $accountId
     * @param int $requestedCount عدد الأكواد المطلوبة
     * @return array ['allowed' => bool, 'max_allowed' => int, 'reason' => string|null]
     */
    public static function checkActivationGenerationLimit(int $superdistId, int $accountId, int $requestedCount = 1): array
    {
        // جلب سجل السوبر موزع + الحساب
        $sa = Db::fetchOne(
            "SELECT * FROM superdist_accounts WHERE superdist_id = ? AND account_id = ?",
            [$superdistId, $accountId]
        );
        
        if (!$sa) {
            return [
                'allowed'     => false,
                'max_allowed' => 0,
                'reason'      => 'no_access',
            ];
        }
        
        // التحقق من انتهاء الصلاحية
        if (strtotime($sa['end_at']) < time()) {
            return [
                'allowed'     => false,
                'max_allowed' => 0,
                'reason'      => 'expired',
            ];
        }
        
        $maxAllowed = $requestedCount;
        $limitHit = null;
        
        // التحقق من الحد اليومي
        if ($sa['activation_limit_daily'] !== null) {
            $remaining = $sa['activation_limit_daily'] - $sa['activations_today'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'daily_limit',
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        // التحقق من الحد الأسبوعي
        if ($sa['activation_limit_weekly'] !== null) {
            $remaining = $sa['activation_limit_weekly'] - $sa['activations_this_week'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'weekly_limit',
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        // التحقق من الحد الشهري
        if ($sa['activation_limit_monthly'] !== null) {
            $remaining = $sa['activation_limit_monthly'] - $sa['activations_this_month'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'monthly_limit',
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        return [
            'allowed'     => $maxAllowed > 0,
            'max_allowed' => max(0, $maxAllowed),
            'reason'      => $maxAllowed < $requestedCount ? 'partial' : null,
        ];
    }
    
    /**
     * زيادة عدادات توليد أكواد التفعيل
     * 
     * @param int $superdistId
     * @param int $accountId
     * @param int $count عدد الأكواد المولّدة
     */
    public static function incrementActivationCounters(int $superdistId, int $accountId, int $count = 1): void
    {
        Db::execute(
            "UPDATE superdist_accounts 
             SET activations_today = activations_today + ?,
                 activations_this_week = activations_this_week + ?,
                 activations_this_month = activations_this_month + ?
             WHERE superdist_id = ? AND account_id = ?",
            [$count, $count, $count, $superdistId, $accountId]
        );
    }
    
    // ================================================================
    // دوال إدارة الحدود (للإدارة)
    // ================================================================
    
    /**
     * تعيين الحدود العامة
     * 
     * @param int $daily
     * @param int $weekly
     * @param int $monthly
     */
    public static function setGlobalLimits(int $daily, int $weekly, int $monthly): void
    {
        Db::execute(
            "UPDATE limits_global SET daily_limit = ?, weekly_limit = ?, monthly_limit = ? WHERE limit_type = 'verification'",
            [$daily, $weekly, $monthly]
        );
    }
    
    /**
     * تعيين حدود سوبر موزع
     * 
     * @param int $superdistId
     * @param int|null $daily
     * @param int|null $weekly
     * @param int|null $monthly
     */
    public static function setSuperDistLimits(int $superdistId, ?int $daily, ?int $weekly, ?int $monthly): void
    {
        $exists = Db::exists('limits_superdist', 'superdist_id = ?', [$superdistId]);
        
        if ($exists) {
            Db::update('limits_superdist', [
                'daily_limit'  => $daily,
                'weekly_limit' => $weekly,
                'monthly_limit'=> $monthly,
            ], 'superdist_id = ?', [$superdistId]);
        } else {
            Db::insert('limits_superdist', [
                'superdist_id'  => $superdistId,
                'daily_limit'   => $daily,
                'weekly_limit'  => $weekly,
                'monthly_limit' => $monthly,
            ]);
        }
    }
    
    /**
     * تعيين حدود مستخدم
     * 
     * @param int $userId
     * @param int|null $daily
     * @param int|null $weekly
     * @param int|null $monthly
     */
    public static function setUserLimits(int $userId, ?int $daily, ?int $weekly, ?int $monthly): void
    {
        $exists = Db::exists('limits_user', 'user_id = ?', [$userId]);
        
        if ($exists) {
            Db::update('limits_user', [
                'daily_limit'  => $daily,
                'weekly_limit' => $weekly,
                'monthly_limit'=> $monthly,
            ], 'user_id = ?', [$userId]);
        } else {
            Db::insert('limits_user', [
                'user_id'       => $userId,
                'daily_limit'   => $daily,
                'weekly_limit'  => $weekly,
                'monthly_limit' => $monthly,
            ]);
        }
    }
    
    /**
     * تعيين حدود مستخدم + حساب
     * 
     * @param int $userId
     * @param int $accountId
     * @param int|null $daily
     * @param int|null $weekly
     * @param int|null $monthly
     */
    public static function setUserAccountLimits(int $userId, int $accountId, ?int $daily, ?int $weekly, ?int $monthly): void
    {
        $exists = Db::exists('limits_user_account', 'user_id = ? AND account_id = ?', [$userId, $accountId]);
        
        if ($exists) {
            Db::update('limits_user_account', [
                'daily_limit'  => $daily,
                'weekly_limit' => $weekly,
                'monthly_limit'=> $monthly,
            ], 'user_id = ? AND account_id = ?', [$userId, $accountId]);
        } else {
            Db::insert('limits_user_account', [
                'user_id'       => $userId,
                'account_id'    => $accountId,
                'daily_limit'   => $daily,
                'weekly_limit'  => $weekly,
                'monthly_limit' => $monthly,
            ]);
        }
    }
    
    /**
     * تعيين حدود حساب (طبقة 3.5)
     * 
     * حدود تنطبق على جميع المستخدمين لهذا الحساب
     * 
     * @param int $accountId
     * @param int|null $daily
     * @param int|null $weekly
     * @param int|null $monthly
     */
    public static function setAccountLimits(int $accountId, ?int $daily, ?int $weekly, ?int $monthly): void
    {
        $exists = Db::exists('limits_account', 'account_id = ?', [$accountId]);
        
        if ($exists) {
            Db::update('limits_account', [
                'daily_limit'  => $daily,
                'weekly_limit' => $weekly,
                'monthly_limit'=> $monthly,
            ], 'account_id = ?', [$accountId]);
        } else {
            Db::insert('limits_account', [
                'account_id'    => $accountId,
                'daily_limit'   => $daily,
                'weekly_limit'  => $weekly,
                'monthly_limit' => $monthly,
            ]);
        }
    }
}
