<?php
/**
 * ============================================================
 * مكتبة توليد الأكواد (Code Generator)
 * ============================================================
 * 
 * توفر دوال لتوليد:
 * - أكواد اشتراك السوبر موزعين (SD-...)
 * - أكواد التفعيل للزبائن (ACT-...)
 * - أكواد التحقق الزمنية (كل 30 ثانية)
 * 
 * الاستخدام:
 *   CodeGenerator::init($config);
 *   $code = CodeGenerator::generateActivationCode();
 *   $verificationCode = CodeGenerator::generateVerificationCode($secretKey);
 */

class CodeGenerator
{
    /** @var array إعدادات النظام */
    private static array $config = [];
    
    /** @var string الأحرف المستخدمة في الأكواد */
    private const CHARSET = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    
    /** @var string أحرف أكواد التحقق (متوافقة مع Steam Guard) */
    private const VERIFICATION_CHARSET = '23456789BCDFGHJKMNPQRTVWXY';
    
    /**
     * تهيئة المكتبة
     * 
     * @param array $config إعدادات من env.php
     */
    public static function init(array $config): void
    {
        self::$config = $config;
    }
    
    /**
     * توليد كود اشتراك سوبر موزع
     * 
     * @return string الكود (مثل: SD-ABCD-1234)
     */
    public static function generateSuperDistCode(): string
    {
        $prefix = self::$config['SUPERDIST_CODE_PREFIX'] ?? 'SD-';
        $length = self::$config['CODE_RANDOM_LENGTH'] ?? 8;
        
        $randomPart = self::generateRandomString($length);
        
        // تنسيق: SD-XXXX-XXXX
        $formatted = substr($randomPart, 0, 4) . '-' . substr($randomPart, 4, 4);
        
        return $prefix . $formatted;
    }
    
    /**
     * توليد كود تفعيل
     * 
     * @return string الكود (مثل: ACT-ABCD-1234)
     */
    public static function generateActivationCode(): string
    {
        $prefix = self::$config['ACTIVATION_CODE_PREFIX'] ?? 'ACT-';
        $length = self::$config['CODE_RANDOM_LENGTH'] ?? 8;
        
        $randomPart = self::generateRandomString($length);
        
        // تنسيق: ACT-XXXX-XXXX
        $formatted = substr($randomPart, 0, 4) . '-' . substr($randomPart, 4, 4);
        
        return $prefix . $formatted;
    }
    
    /**
     * توليد مجموعة أكواد تفعيل
     * 
     * @param int $count العدد المطلوب
     * @return array مصفوفة الأكواد
     */
    public static function generateActivationCodes(int $count): array
    {
        $codes = [];
        $attempts = 0;
        $maxAttempts = $count * 10; // لتجنب التكرار اللانهائي
        
        while (count($codes) < $count && $attempts < $maxAttempts) {
            $code = self::generateActivationCode();
            if (!in_array($code, $codes, true)) {
                $codes[] = $code;
            }
            $attempts++;
        }
        
        return $codes;
    }
    
    /**
     * توليد كود تحقق زمني (TOTP-like)
     * 
     * ============================================================
     * المواصفات:
     * ============================================================
     * - طول الكود: 5 أحرف
     * - الفاصل الزمني: 30 ثانية
     * - مجموعة الأحرف: 23456789BCDFGHJKMNPQRTVWXY (26 حرف)
     * - الخوارزمية: HMAC-SHA1 (متوافقة مع Steam Guard)
     * 
     * ============================================================
     * التوافق مع الموقع:
     * ============================================================
     * لكي يعمل الكود بشكل صحيح، يجب أن يستخدم الموقع:
     * - نفس المفتاح السري (secret_key)
     * - نفس الفاصل الزمني (30 ثانية)
     * - نفس الخوارزمية (HMAC-SHA1 + نفس طريقة الاستخراج)
     * 
     * ============================================================
     * 
     * @param string $secretKey المفتاح السري (base64 أو نص عادي)
     * @param int|null $timestamp الوقت بالثواني (Unix timestamp)
     *                           - null = استخدام الوقت الحالي time()
     *                           - int = وقت محدد (للاختبار أو التحقق اليدوي)
     * @return string كود التحقق (5 أحرف بالضبط)
     * 
     * @example
     *   // توليد كود للوقت الحالي
     *   $code = CodeGenerator::generateVerificationCode($secretKey);
     *   
     *   // توليد كود لوقت محدد (للاختبار)
     *   $code = CodeGenerator::generateVerificationCode($secretKey, 1701388800);
     */
    public static function generateVerificationCode(string $secretKey, ?int $timestamp = null): string
    {
        // محاولة فك تشفير base64، إذا فشل استخدم النص كما هو
        $secret = base64_decode($secretKey, true);
        if ($secret === false) {
            $secret = $secretKey;
        }
        
        // حساب الفترة الزمنية (كل 30 ثانية)
        $time = $timestamp ?? time();
        $counter = (int) floor($time / 30);
        
        // تحويل العداد إلى bytes (big-endian)
        $counterBytes = pack('N*', 0) . pack('N*', $counter);
        
        // حساب HMAC-SHA1
        $hmac = hash_hmac('sha1', $counterBytes, $secret, true);
        
        // استخراج الـ offset
        $offset = ord(substr($hmac, -1)) & 0x0F;
        
        // استخراج 4 bytes وتحويلها إلى رقم
        $otp = (ord($hmac[$offset]) & 0x7F) << 24 |
               (ord($hmac[$offset + 1]) & 0xFF) << 16 |
               (ord($hmac[$offset + 2]) & 0xFF) << 8 |
               (ord($hmac[$offset + 3]) & 0xFF);
        
        // تحويل الرقم إلى 5 أحرف
        $chars = self::VERIFICATION_CHARSET;
        $code = '';
        for ($i = 0; $i < 5; $i++) {
            $code .= $chars[$otp % strlen($chars)];
            $otp = intdiv($otp, strlen($chars));
        }
        
        return $code;
    }
    
    /**
     * التحقق من كود تحقق
     * 
     * يتحقق من الكود للفترة الحالية والسابقة (للتسامح مع التأخير)
     * 
     * @param string $code الكود المُدخل
     * @param string $secretKey المفتاح السري
     * @param int $tolerance عدد الفترات المسموحة (افتراضي: 1)
     * @return bool
     */
    public static function verifyVerificationCode(string $code, string $secretKey, int $tolerance = 1): bool
    {
        $time = time();
        $code = strtoupper(trim($code));
        
        for ($i = -$tolerance; $i <= $tolerance; $i++) {
            $checkTime = $time + ($i * 30);
            $expectedCode = self::generateVerificationCode($secretKey, $checkTime);
            
            if (hash_equals($expectedCode, $code)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * توليد سلسلة عشوائية
     * 
     * @param int $length الطول
     * @return string
     */
    private static function generateRandomString(int $length): string
    {
        $charset = self::CHARSET;
        $charsetLength = strlen($charset);
        $result = '';
        
        // استخدام random_int للأمان
        for ($i = 0; $i < $length; $i++) {
            $result .= $charset[random_int(0, $charsetLength - 1)];
        }
        
        return $result;
    }
    
    /**
     * تحديد نوع الكود من البادئة
     * 
     * @param string $code الكود
     * @return string|null 'superdist' أو 'activation' أو null
     */
    public static function getCodeType(string $code): ?string
    {
        $code = strtoupper(trim($code));
        
        $sdPrefix = strtoupper(self::$config['SUPERDIST_CODE_PREFIX'] ?? 'SD-');
        $actPrefix = strtoupper(self::$config['ACTIVATION_CODE_PREFIX'] ?? 'ACT-');
        
        if (strpos($code, $sdPrefix) === 0) {
            return 'superdist';
        }
        
        if (strpos($code, $actPrefix) === 0) {
            return 'activation';
        }
        
        return null;
    }
    
    /**
     * التحقق من صيغة الكود
     * 
     * @param string $code الكود
     * @param string $type نوع الكود: 'superdist' أو 'activation'
     * @return bool
     */
    public static function isValidCodeFormat(string $code, string $type): bool
    {
        $code = strtoupper(trim($code));
        
        if ($type === 'superdist') {
            // صيغة: SD-XXXX-XXXX
            return (bool) preg_match('/^SD-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $code);
        }
        
        if ($type === 'activation') {
            // صيغة: ACT-XXXX-XXXX
            return (bool) preg_match('/^ACT-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $code);
        }
        
        return false;
    }
    
    /**
     * حساب الثواني المتبقية لكود التحقق الحالي
     * 
     * @return int الثواني المتبقية (0-29)
     */
    public static function getVerificationCodeRemainingSeconds(): int
    {
        return 30 - (time() % 30);
    }
    
    /**
     * توليد معرف فريد (UUID v4)
     * 
     * @return string
     */
    public static function generateUUID(): string
    {
        $data = random_bytes(16);
        
        // Set version to 0100
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        // Set bits 6-7 to 10
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
}
