<?php
/**
 * مدير صلاحيات الوصول الدفعي
 * Batch Access Manager
 * 
 * يدير تغيير صلاحيات الوصول للزبائن دفعة واحدة
 */

class BatchAccessManager
{
    /**
     * معاينة تغيير صلاحيات الوصول
     */
    public static function previewUsersAccessChange(array $accountIds, string $newAccessType, int $newAccessDays = null, bool $force = false): array
    {
        $accountIdsStr = implode(',', $accountIds);
        
        $query = "SELECT 
            COUNT(*) as total_affected,
            SUM(CASE WHEN ua.access_type = 'lifetime' THEN 1 ELSE 0 END) as lifetime_count,
            SUM(CASE WHEN ua.access_type = 'timed' THEN 1 ELSE 0 END) as timed_count,
            COUNT(DISTINCT ua.user_id) as unique_users,
            COUNT(DISTINCT ua.account_id) as unique_accounts
        FROM user_accounts ua
        WHERE ua.account_id IN ({$accountIdsStr})";
        
        if (!$force) {
            $query .= " AND ua.access_type != 'lifetime'";
        }
        
        $stats = Db::fetchOne($query);
        
        // جلب عينة من السجلات مع حساب الأيام المتبقية
        $sampleQuery = "SELECT ua.*, u.telegram_id, a.name as account_name, a.slug,
            DATEDIFF(NOW(), COALESCE(ua.activated_at, ua.created_at)) as days_used
        FROM user_accounts ua
        JOIN users u ON ua.user_id = u.id
        JOIN accounts a ON ua.account_id = a.id
        WHERE ua.account_id IN ({$accountIdsStr})";
        
        if (!$force) {
            $sampleQuery .= " AND ua.access_type != 'lifetime'";
        }
        
        $sampleQuery .= " LIMIT 10";
        
        $samples = Db::fetchAll($sampleQuery);
        
        // حساب عدد الاشتراكات التي ستنتهي فوراً (إذا كان التغيير لـ timed)
        $willExpireImmediately = 0;
        $willHaveRemaining = 0;
        
        if ($newAccessType === 'timed' && $newAccessDays > 0) {
            $expireQuery = "SELECT 
                SUM(CASE WHEN DATEDIFF(NOW(), COALESCE(ua.activated_at, ua.created_at)) >= {$newAccessDays} THEN 1 ELSE 0 END) as will_expire,
                SUM(CASE WHEN DATEDIFF(NOW(), COALESCE(ua.activated_at, ua.created_at)) < {$newAccessDays} THEN 1 ELSE 0 END) as will_remain
            FROM user_accounts ua
            WHERE ua.account_id IN ({$accountIdsStr})";
            
            if (!$force) {
                $expireQuery .= " AND ua.access_type != 'lifetime'";
            }
            
            $expireStats = Db::fetchOne($expireQuery);
            $willExpireImmediately = (int) ($expireStats['will_expire'] ?? 0);
            $willHaveRemaining = (int) ($expireStats['will_remain'] ?? 0);
        }
        
        return [
            'stats' => $stats,
            'samples' => $samples,
            'conflicts' => $force ? 0 : ($stats['lifetime_count'] ?? 0),
            'estimated_time_seconds' => ceil(($stats['total_affected'] ?? 0) / 100),
            'will_expire_immediately' => $willExpireImmediately,
            'will_have_remaining' => $willHaveRemaining,
            'new_access_days' => $newAccessDays,
        ];
    }
    
    /**
     * تنفيذ تغيير صلاحيات الوصول
     */
    public static function executeUsersAccessChange(
        array $accountIds,
        string $newAccessType,
        ?int $newAccessDays,
        bool $force,
        int $adminId,
        int $adminTelegramId,
        bool $notify = false
    ): array {
        $startTime = microtime(true);
        $accountIdsStr = implode(',', $accountIds);
        
        try {
            Db::beginTransaction();
            
            // قفل السجلات
            $query = "SELECT ua.* FROM user_accounts ua
            WHERE ua.account_id IN ({$accountIdsStr})";
            
            if (!$force) {
                $query .= " AND ua.access_type != 'lifetime'";
            }
            
            $query .= " FOR UPDATE";
            
            $records = Db::fetchAll($query);
            
            if (empty($records)) {
                Db::rollback();
                return [
                    'success' => false,
                    'message' => 'لا توجد سجلات للتحديث',
                    'affected_count' => 0,
                ];
            }
            
            $affected = 0;
            $errors = [];
            
            foreach ($records as $record) {
                try {
                    $updates = [
                        'access_type' => $newAccessType,
                    ];
                    
                    if ($newAccessType === 'timed') {
                        $updates['access_days'] = $newAccessDays;
                        
                        // حساب تاريخ الانتهاء من تاريخ التفعيل الأصلي (استقطاع المدة المستخدمة)
                        $activatedAt = $record['activated_at'] ?? $record['created_at'];
                        $expiresAt = strtotime($activatedAt . " +{$newAccessDays} days");
                        
                        // إذا كان تاريخ الانتهاء المحسوب في الماضي، يعني الاشتراك منتهي
                        if ($expiresAt < time()) {
                            // الاشتراك منتهي - نضعه كمنتهي الآن
                            $updates['access_expires_at'] = date('Y-m-d H:i:s', time());
                            $updates['status'] = 'expired';
                        } else {
                            $updates['access_expires_at'] = date('Y-m-d H:i:s', $expiresAt);
                        }
                    } else {
                        // lifetime - لا يوجد انتهاء
                        $updates['access_days'] = null;
                        $updates['access_expires_at'] = null;
                    }
                    
                    Db::update('user_accounts', $updates, 'id = ?', [$record['id']]);
                    $affected++;
                    
                } catch (Throwable $e) {
                    $errors[] = [
                        'user_id' => $record['user_id'],
                        'account_id' => $record['account_id'],
                        'error' => $e->getMessage(),
                    ];
                }
            }
            
            // تسجيل في admin_audit (اختياري - يتجاهل الخطأ إذا لم يكن الجدول موجوداً)
            $executionTime = round((microtime(true) - $startTime) * 1000);
            
            try {
                Db::insert('admin_audit', [
                    'admin_id' => $adminId,
                    'admin_telegram_id' => $adminTelegramId,
                    'action' => 'set_users_access',
                    'target_type' => 'user_accounts',
                    'target_ids' => json_encode($accountIds),
                    'old_values' => json_encode(['note' => 'bulk_update']),
                    'new_values' => json_encode([
                        'access_type' => $newAccessType,
                        'access_days' => $newAccessDays,
                    ]),
                    'affected_count' => $affected,
                    'is_preview' => 0,
                    'flags' => json_encode(['force' => $force, 'notify' => $notify]),
                    'status' => empty($errors) ? 'success' : 'partial',
                    'error_message' => empty($errors) ? null : json_encode($errors),
                    'execution_time_ms' => $executionTime,
                ]);
            } catch (Throwable $e) {
                // تجاهل أخطاء التدقيق - الجدول قد لا يكون موجوداً بعد
            }
            
            Db::commit();
            
            // إرسال إشعارات للزبائن إذا طُلب ذلك
            if ($notify && $affected > 0) {
                self::notifyUsersAboutAccessChange($records, $newAccessType, $newAccessDays);
            }
            
            return [
                'success' => true,
                'affected_count' => $affected,
                'errors_count' => count($errors),
                'errors' => $errors,
                'execution_time_ms' => $executionTime,
            ];
            
        } catch (Throwable $e) {
            Db::rollback();
            return [
                'success' => false,
                'message' => $e->getMessage(),
                'affected_count' => 0,
            ];
        }
    }
    
    /**
     * إرسال إشعارات للزبائن
     */
    private static function notifyUsersAboutAccessChange(array $records, string $newAccessType, ?int $newAccessDays): void
    {
        // تجميع الزبائن الفريدين
        $users = [];
        foreach ($records as $record) {
            $userId = $record['user_id'];
            if (!isset($users[$userId])) {
                $user = Db::fetchOne("SELECT * FROM users WHERE id = ?", [$userId]);
                if ($user) {
                    $users[$userId] = $user;
                }
            }
        }
        
        // إرسال الإشعارات
        foreach ($users as $user) {
            $text = "📢 <b>تحديث صلاحيات الوصول</b>\n\n";
            
            if ($newAccessType === 'lifetime') {
                $text .= "تم تحديث اشتراكك إلى: ♾️ <b>مدى الحياة</b>\n\n";
                $text .= "✅ يمكنك الآن الوصول بشكل دائم!";
            } else {
                $text .= "تم تحديث اشتراكك إلى: ⏱️ <b>{$newAccessDays} يوم</b>\n\n";
                $text .= "📅 سينتهي اشتراكك بعد {$newAccessDays} يوم من الآن.";
            }
            
            try {
                Telegram::sendMessage($user['telegram_id'], $text);
            } catch (Throwable $e) {
                // تجاهل أخطاء الإرسال
            }
        }
    }
    
    /**
     * ضبط الوصول الافتراضي للحساب
     */
    public static function setDefaultAccountAccess(
        int $accountId,
        ?string $accessType,
        ?int $accessDays,
        bool $inheritGlobal,
        int $adminId,
        int $adminTelegramId
    ): bool {
        try {
            $account = Db::fetchOne("SELECT * FROM accounts WHERE id = ?", [$accountId]);
            if (!$account) {
                return false;
            }
            
            $oldValues = [
                'default_access_type' => $account['default_access_type'],
                'default_access_days' => $account['default_access_days'],
                'default_inherit_global' => $account['default_inherit_global'],
            ];
            
            $updates = [
                'default_inherit_global' => $inheritGlobal ? 1 : 0,
            ];
            
            if (!$inheritGlobal) {
                $updates['default_access_type'] = $accessType;
                $updates['default_access_days'] = $accessType === 'timed' ? $accessDays : null;
            }
            
            Db::update('accounts', $updates, 'id = ?', [$accountId]);
            
            // تسجيل في admin_audit
            Db::insert('admin_audit', [
                'admin_id' => $adminId,
                'admin_telegram_id' => $adminTelegramId,
                'action' => 'set_default_account_access',
                'target_type' => 'accounts',
                'target_ids' => json_encode([$accountId]),
                'old_values' => json_encode($oldValues),
                'new_values' => json_encode($updates),
                'affected_count' => 1,
                'is_preview' => 0,
                'status' => 'success',
            ]);
            
            return true;
            
        } catch (Throwable $e) {
            Logger::error('Failed to set default account access', [
                'account_id' => $accountId,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }
    
    /**
     * الحصول على الوصول الافتراضي للحساب
     */
    public static function getDefaultAccountAccess(int $accountId): ?array
    {
        $account = Db::fetchOne(
            "SELECT id, slug, name, default_access_type, default_access_days, default_inherit_global 
             FROM accounts WHERE id = ?",
            [$accountId]
        );
        
        if (!$account) {
            return null;
        }
        
        if ($account['default_inherit_global']) {
            // جلب الإعدادات العامة
            $globalType = Db::fetchValue(
                "SELECT setting_value FROM global_settings WHERE setting_key = 'default_access_type'"
            ) ?: 'lifetime';
            
            $globalDays = (int) Db::fetchValue(
                "SELECT setting_value FROM global_settings WHERE setting_key = 'default_access_days'"
            ) ?: 30;
            
            $account['effective_access_type'] = $globalType;
            $account['effective_access_days'] = $globalDays;
            $account['source'] = 'global';
        } else {
            $account['effective_access_type'] = $account['default_access_type'];
            $account['effective_access_days'] = $account['default_access_days'];
            $account['source'] = 'account';
        }
        
        return $account;
    }
    
    /**
     * قائمة الحسابات التي ترث الإعدادات العامة
     */
    public static function listAccountsWithDefaultInherit(int $page = 1, int $perPage = 20): array
    {
        $offset = ($page - 1) * $perPage;
        
        $total = (int) Db::fetchValue(
            "SELECT COUNT(*) FROM accounts WHERE default_inherit_global = 1"
        );
        
        $accounts = Db::fetchAll(
            "SELECT a.id, a.slug, a.name, a.default_inherit_global,
                    COUNT(DISTINCT ua.user_id) as active_users_count
             FROM accounts a
             LEFT JOIN user_accounts ua ON a.id = ua.account_id
             WHERE a.default_inherit_global = 1
             GROUP BY a.id, a.slug, a.name, a.default_inherit_global
             ORDER BY a.name ASC
             LIMIT ? OFFSET ?",
            [$perPage, $offset]
        );
        
        return [
            'total' => $total,
            'page' => $page,
            'per_page' => $perPage,
            'total_pages' => ceil($total / $perPage),
            'accounts' => $accounts,
        ];
    }
}
