<?php
/**
 * ============================================================
 * مهمة يومية - تصفير العدادات اليومية
 * ============================================================
 * 
 * تُنفذ: يومياً عند منتصف الليل (00:00)
 * 
 * Cron Command (cPanel):
 *   0 0 * * * /usr/bin/php /home/username/public_html/cron/daily_reset.php
 * 
 * الوظائف:
 *   1. تصفير عدادات توليد الأكواد اليومية
 *   2. تحديث حالة الأكواد المنتهية
 *   3. تحديث حالة اشتراكات المستخدمين المنتهية
 *   4. تنظيف user_states القديمة
 */

// منع التنفيذ عبر المتصفح
if (php_sapi_name() !== 'cli' && !defined('CRON_ALLOWED')) {
    http_response_code(403);
    exit('CLI only');
}

define('BOT_STARTED', true);
define('BOT_TYPE', 'cron');

require __DIR__ . '/../libs/bootstrap.php';

$startTime = microtime(true);
$log = [];

echo "=== Daily Reset Cron Job ===\n";
echo "Started at: " . date('Y-m-d H:i:s') . "\n\n";

try {
    // ============================================================
    // 1. تصفير عدادات توليد الأكواد اليومية (Idempotent)
    // ============================================================
    echo "1. Resetting daily activation counters...\n";
    
    $today = date('Y-m-d');
    
    // التصفير فقط للسجلات التي لم تُصفَّر اليوم
    $affected = Db::execute(
        "UPDATE superdist_accounts 
         SET activations_today = 0,
             last_daily_reset = ?
         WHERE last_daily_reset IS NULL OR last_daily_reset < ?",
        [$today, $today]
    );
    
    echo "   - Reset {$affected} records\n";
    $log[] = "Daily counters reset: {$affected}";
    
    // ============================================================
    // 2. تحديث حالة الأكواد المنتهية
    // ============================================================
    echo "2. Expiring old activation codes...\n";
    
    $affected = Db::execute(
        "UPDATE activation_codes 
         SET status = 'expired' 
         WHERE status = 'unused' 
           AND expires_at IS NOT NULL 
           AND expires_at < NOW()"
    );
    
    echo "   - Expired {$affected} codes\n";
    $log[] = "Expired codes: {$affected}";
    
    // ============================================================
    // 3. تحديث حالة اشتراكات المستخدمين المنتهية
    // ============================================================
    echo "3. Updating expired user subscriptions...\n";
    
    $affected = Db::execute(
        "UPDATE user_accounts 
         SET status = 'expired' 
         WHERE status = 'active' 
           AND access_type = 'timed'
           AND access_expires_at IS NOT NULL 
           AND access_expires_at < NOW()"
    );
    
    echo "   - Expired {$affected} user subscriptions\n";
    $log[] = "Expired user subscriptions: {$affected}";
    
    // ============================================================
    // 4. تنظيف user_states القديمة
    // ============================================================
    echo "4. Cleaning old user states...\n";
    
    $affected = State::cleanupExpired();
    
    echo "   - Cleaned {$affected} states\n";
    $log[] = "Cleaned states: {$affected}";
    
    // ============================================================
    // تسجيل الحدث
    // ============================================================
    Logger::event('cron_executed', 'system', null, null, null, [
        'job'     => 'daily_reset',
        'results' => $log,
    ]);
    
} catch (Throwable $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    Logger::error('Daily cron failed', ['error' => $e->getMessage()]);
}

$duration = round((microtime(true) - $startTime) * 1000, 2);
echo "\nCompleted in {$duration}ms\n";
