<?php
/**
 * ============================================================
 * مهمة يومية - تنظيف عام
 * ============================================================
 * 
 * تُنفذ: يومياً عند الساعة 03:00
 * 
 * Cron Command (cPanel):
 *   0 3 * * * /usr/bin/php /home/username/public_html/cron/cleanup.php
 * 
 * الوظائف:
 *   1. حذف ملفات الأكواد المؤقتة القديمة
 *   2. أرشفة سجلات الأحداث القديمة (اختياري)
 *   3. تنظيف جدول verification_logs القديم (اختياري)
 */

if (php_sapi_name() !== 'cli' && !defined('CRON_ALLOWED')) {
    http_response_code(403);
    exit('CLI only');
}

define('BOT_STARTED', true);
define('BOT_TYPE', 'cron');

require __DIR__ . '/../libs/bootstrap.php';

$startTime = microtime(true);
$log = [];

echo "=== Cleanup Cron Job ===\n";
echo "Started at: " . date('Y-m-d H:i:s') . "\n\n";

try {
    // ============================================================
    // 1. حذف ملفات temp القديمة (أكثر من يوم)
    // ============================================================
    echo "1. Cleaning temp files...\n";
    
    $tempPath = config('TEMP_PATH', __DIR__ . '/../temp/');
    $deletedFiles = 0;
    
    if (is_dir($tempPath)) {
        $files = glob($tempPath . '*');
        $oneDayAgo = time() - 86400;
        
        foreach ($files as $file) {
            if (is_file($file) && filemtime($file) < $oneDayAgo) {
                if (basename($file) !== '.gitkeep') {
                    unlink($file);
                    $deletedFiles++;
                }
            }
        }
    }
    
    echo "   - Deleted {$deletedFiles} temp files\n";
    $log[] = "Deleted temp files: {$deletedFiles}";
    
    // ============================================================
    // 2. حذف سجلات الأحداث القديمة (أكثر من 90 يوم)
    // ============================================================
    echo "2. Archiving old event logs...\n";
    
    $affected = Db::execute(
        "DELETE FROM event_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)"
    );
    
    echo "   - Archived {$affected} old events\n";
    $log[] = "Archived events: {$affected}";
    
    // ============================================================
    // 3. حذف سجلات التحقق القديمة (أكثر من 30 يوم)
    // ============================================================
    echo "3. Cleaning old verification logs...\n";
    
    $affected = Db::execute(
        "DELETE FROM verification_logs WHERE requested_at < DATE_SUB(NOW(), INTERVAL 30 DAY)"
    );
    
    echo "   - Cleaned {$affected} old verification logs\n";
    $log[] = "Cleaned verification logs: {$affected}";
    
    // ============================================================
    // 4. تنظيف ملفات السجلات القديمة
    // ============================================================
    echo "4. Cleaning old log files...\n";
    
    $logPath = config('LOG_PATH', __DIR__ . '/../logs/');
    $deletedLogs = 0;
    $thirtyDaysAgo = time() - (30 * 86400);
    
    if (is_dir($logPath)) {
        $files = glob($logPath . '*.log');
        
        foreach ($files as $file) {
            if (is_file($file) && filemtime($file) < $thirtyDaysAgo) {
                unlink($file);
                $deletedLogs++;
            }
        }
    }
    
    echo "   - Deleted {$deletedLogs} old log files\n";
    $log[] = "Deleted log files: {$deletedLogs}";
    
    // ============================================================
    // تسجيل الحدث
    // ============================================================
    Logger::event('cron_executed', 'system', null, null, null, [
        'job'     => 'cleanup',
        'results' => $log,
    ]);
    
} catch (Throwable $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    Logger::error('Cleanup cron failed', ['error' => $e->getMessage()]);
}

$duration = round((microtime(true) - $startTime) * 1000, 2);
echo "\nCompleted in {$duration}ms\n";
